package com.elitesland.cloudt.context.spi.support;

import com.elitesland.cloudt.context.SpringContextHolder;
import com.elitesland.yst.common.annotation.context.spi.Spi;
import lombok.extern.log4j.Log4j2;
import org.springframework.context.annotation.Primary;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.OrderUtils;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.util.Objects;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/16
 */
@Log4j2
class ProviderWrapper<T> {

    /**
     * 实例对象
     */
    private final T provider;
    /**
     * 接口注解
     */
    private final Spi spi;
    /**
     * 类型
     */
    private final Class<?> clazz;
    /**
     * 实例来源
     */
    private Source source = Source.LIB;
    /**
     * 顺序
     */
    private int order = 0;
    /**
     * 是否是默认实例
     */
    private boolean primary = false;

    ProviderWrapper(@NonNull T provider, Spi spi) {
        this.provider = provider;
        this.spi = spi;
        this.clazz = provider.getClass();

        analyze();
    }

    private void analyze() {
        // 分析来源
        analyzeSource();

        // 分析顺序
        analyzeOrder();

        // 分析是否是缺省的实例
        analyzePrimary();

        // 装配依赖
        autowiredBean();
    }

    private void analyzeSource() {
        var path = Objects.requireNonNull(clazz.getResource("/")).toString();
        var sourceType = "/classes/";
        if (path.contains(sourceType)) {
            this.source = Source.CLASSES;
            return;
        }
    }

    private void analyzeOrder() {
        Integer sort = obtainOrderForSpring();
        if (sort != null) {
            this.order = sort;
            return;
        }

        // 默认如果是classes内的，则顺序为1，比lib（lib中的是0）中的优先级高
        if (this.source == Source.CLASSES) {
            this.order = 1;
        }
    }

    private void analyzePrimary() {
        if (spi != null && StringUtils.hasText(spi.primary())) {
            // 接口上有标注primary，则使用接口的
            return;
        }

        var annoPrimary = SpiUtil.obtainAnnotationSingle(clazz, Primary.class);
        this.primary = annoPrimary != null;
    }

    private void autowiredBean() {
        try {
            SpringContextHolder.autowireBean(this.provider);
        } catch (Exception e) {
            log.warn("装配bean【" + this.clazz.getName() + "】依赖失败：", e);
        }
    }

    /**
     * 获取spring相关排序的注解
     * <p>
     * 优先级依次是：org.springframework.core.annotation.Order、javax.annotation.Priority、org.springframework.core.Ordered
     *
     * @return 顺序
     */
    private Integer obtainOrderForSpring() {
        var sort = OrderUtils.getOrder(this.clazz);
        if (sort != null) {
            return sort;
        }

        if (Ordered.class.isAssignableFrom(this.clazz)) {
            return ((Ordered) provider).getOrder();
        }

        return null;
    }

    public Object getProvider() {
        return provider;
    }

    public Class<?> getClazz() {
        return clazz;
    }

    public Source getSource() {
        return source;
    }

    public int getOrder() {
        return order;
    }

    public boolean isPrimary() {
        return primary;
    }

    public void setPrimary(boolean primary) {
        this.primary = primary;
    }
}
