package com.elitesland.cloudt.context.threadpool;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.dtp.common.config.ThreadPoolProperties;
import com.dtp.core.DtpRegistry;
import com.dtp.core.support.TaskWrapper;
import com.dtp.core.thread.DtpExecutor;
import com.dtp.core.thread.NamedThreadFactory;
import com.dtp.core.thread.ThreadPoolBuilder;
import com.elitesland.cloudt.context.threadpool.support.ContextTransferDelegate;
import lombok.extern.log4j.Log4j2;

import java.util.Objects;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 线程池工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/2/21
 */
@Log4j2
public final class ThreadPoolHolder {

    private static final String THREAD_POOL_NAME = "cloudT-dtp-common";
    private static final String THREAD_PREFIX = "cloudT-common-";
    private static final int THREAD_MAX_SIZE = Runtime.getRuntime().availableProcessors();

    /**
     * 是否启用动态线程池管理
     */
    private static boolean dtpEnabled = false;

    private ThreadPoolHolder() {
    }

    /**
     * 创建线程池
     *
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool() {
        return createThreadPool(null, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix) {
        return createThreadPool(threadPrefix, null, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @return 线程池
     */
    public static ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize) {
        return createThreadPool(threadPrefix, coreSize, null);
    }

    /**
     * 创建线程池
     *
     * @param threadPrefix 线程名称前缀
     * @param coreSize     线程最小数量
     * @param maxSize      线程最多数量
     * @return 线程池
     */
    public static synchronized ThreadPoolExecutor createThreadPool(String threadPrefix, Integer coreSize, Integer maxSize) {
        return createThreadPool(null, threadPrefix, coreSize, maxSize);
    }

    /**
     * 创建线程池
     *
     * @param threadPoolName 线程池名称
     * @param threadPrefix   线程名称前缀
     * @param coreSize       线程最小数量
     * @param maxSize        线程最多数量
     * @return 线程池
     */
    public static synchronized ThreadPoolExecutor createThreadPool(String threadPoolName, String threadPrefix, Integer coreSize, Integer maxSize) {
        threadPoolName = CharSequenceUtil.blankToDefault(threadPoolName, THREAD_POOL_NAME);
        threadPrefix = CharSequenceUtil.blankToDefault(threadPrefix, THREAD_PREFIX);
        coreSize = ObjectUtil.defaultIfNull(coreSize, 1);
        maxSize = ObjectUtil.defaultIfNull(maxSize, THREAD_MAX_SIZE);

        DtpExecutor executor = null;
        if (existsRegistry(threadPoolName)) {
            executor = DtpRegistry.getDtpExecutor(threadPoolName);
        }

        ThreadPoolProperties propUpdate = null;
        if (executor == null) {
            executor = ThreadPoolBuilder.newBuilder()
                    .threadPoolName(threadPoolName)
                    .threadFactory(threadPrefix)
                    .corePoolSize(Objects.requireNonNullElse(coreSize, 1))
                    .maximumPoolSize(Objects.requireNonNullElse(maxSize, THREAD_MAX_SIZE))
                    .taskWrapper(taskWrapper())
                    .buildDynamic();
        } else {
            propUpdate = new ThreadPoolProperties();
            propUpdate.setCorePoolSize(Objects.requireNonNullElse(coreSize, 1));
            propUpdate.setMaximumPoolSize(Objects.requireNonNullElse(maxSize, THREAD_MAX_SIZE));
            propUpdate.setThreadNamePrefix(threadPrefix);
        }

        if (dtpEnabled) {
            if (propUpdate == null) {
                DtpRegistry.registerDtp(executor, "cloudt-common");
            } else {
                executor.setCorePoolSize(propUpdate.getCorePoolSize());
                executor.setMaximumPoolSize(propUpdate.getMaximumPoolSize());
                executor.setThreadFactory(new NamedThreadFactory(propUpdate.getThreadNamePrefix()));
            }
        }
        return executor;
    }

    static void setDtpEnabled(boolean dtpEnabled) {
        ThreadPoolHolder.dtpEnabled = dtpEnabled;
    }

    private static boolean existsRegistry(String threadPoolName) {
        if (!dtpEnabled) {
            return false;
        }
        return DtpRegistry.listAllDtpNames().contains(threadPoolName);
    }

    private static TaskWrapper taskWrapper() {
        return runnable -> {
            var contextTransferDelegate = ContextTransferDelegate.getInstance();
            // 先获取上下文信息
            contextTransferDelegate.getContext();

            return () -> {
                // 设置上下文信息
                contextTransferDelegate.setContext();

                try {
                    runnable.run();
                } finally {
                    // 清除上下文信息
                    contextTransferDelegate.clearContext();
                }
            };
        };
    }
}

