package com.elitesland.cloudt.tenant;

import com.elitesland.cloudt.context.SpringContextHolder;
import com.elitesland.cloudt.tenant.config.datasource.TenantSession;
import com.elitesland.cloudt.tenant.config.support.TenantContextHolder;
import com.elitesland.cloudt.tenant.provider.TenantProvider;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;

/**
 * 租户客户端工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/6
 */
@Log4j2
public class TenantClient {

    private static TenantProvider tenantProvider;

    private TenantClient() {
    }

    /**
     * 获取当前租户信息
     *
     * @return 租户信息
     */
    public static SysTenantDTO getCurrentTenant() {
        return TenantContextHolder.getCurrentTenant();
    }

    /**
     * 获取租户信息
     *
     * @param tenantId 租户ID
     * @return 租户信息
     */
    public static SysTenantDTO getTenant(Long tenantId) {
        if (tenantId == null) {
            return null;
        }
        return getTenantProvider().getById(tenantId).get();
    }

    /**
     * 将事务切换至指定租户
     *
     * @param tenantId 租户ID
     */
    public static void switchTraditionalForTenant(Long tenantId) {
        Assert.notNull(tenantId, "租户ID为空");

        var tenant = getTenantProvider().getById(tenantId);
        Assert.isTrue(tenant.isPresent(), "租户不存在");

        TenantSession.setCurrentTenant(tenant.get());
        TenantSession.clearUseDefaultSchema();
    }

    /**
     * 将事务切换至当前租户
     *
     */
    public static void switchTenantForTenantCurrent() {
        TenantSession.clearCurrent();
        TenantSession.clearUseDefaultSchema();
    }

    /**
     * 将事务切换至默认租户
     *
     */
    public static void switchTenantForTenantDefault() {
        TenantSession.clearCurrent();
        TenantSession.setUseDefaultSchema();
    }

    private static TenantProvider getTenantProvider() {
        if (tenantProvider == null) {
            tenantProvider = SpringContextHolder.getBean(TenantProvider.class);
        }
        return tenantProvider;
    }
}
