package com.elitesland.cloudt.tenant.config.datasource;

import lombok.extern.log4j.Log4j2;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;

/**
 * 数据源提供者的父类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/25
 */
@Log4j2
public abstract class AbstractTenantDatasourceProvider {

    /**
     * 默认schema
     */
    private static String defaultSchema;

    /**
     * 默认数据源
     */
    private static DataSource defaultDataSource;

    /**
     * 数据库类型
     */
    private static Database databaseType;

    private AbstractTenantDatasourceProvider() {
    }

    /**
     * 获取默认schema
     *
     * @return 默认schema
     */
    public static String getDefaultSchema() {
        return defaultSchema;
    }

    /**
     * 设置默认schemaName
     *
     * @param defaultSchema 默认schema名称
     */
    static void setDefaultSchema(String defaultSchema) {
        if (StringUtils.hasText(AbstractTenantDatasourceProvider.defaultSchema)
                || !StringUtils.hasText(defaultSchema)) {
            return;
        }

        AbstractTenantDatasourceProvider.defaultSchema = defaultSchema;
    }

    /**
     * 获取数据源
     *
     * @return 数据源
     */
    public static DataSource getDefaultDataSource() {
        return defaultDataSource;
    }

    /**
     * 设置默认数据源
     *
     * @param defaultDataSource 数据源
     */
    static void setDefaultDataSource(DataSource defaultDataSource) {
        if (defaultDataSource == null) {
            return;
        }
        AbstractTenantDatasourceProvider.defaultDataSource = defaultDataSource;
    }

    public static Database getDatabaseType() {
        return databaseType;
    }

    /**
     * 设置默认数据库类型
     *
     * @param databaseType 数据库类型
     */
    static void setDatabaseType(Database databaseType) {
        if (databaseType == null || AbstractTenantDatasourceProvider.databaseType != null) {
            return;
        }
        AbstractTenantDatasourceProvider.databaseType = databaseType;
    }

    /**
     * 根据数据库连接解析数据库类型
     *
     * @param url 数据库连接
     * @return 数据库类型
     */
    public static Database getDatabaseTypeByUrl(String url) {
        if (url == null) {
            return null;
        }
        int index = url.indexOf("://");

        String temp;
        if (index > 0) {
            temp = url.substring(0, index);
        } else {
            return null;
        }

        index = temp.lastIndexOf(":");
        if (index > 0) {
            temp = temp.substring(index + 1);
        }
        try {
            return Database.valueOf(temp.toUpperCase());
        } catch (IllegalArgumentException e) {
            log.error("解析数据库类型失败：{}", url);
            return null;
        }
    }

    /**
     * 生成切换schema的SQL
     *
     * @param databaseType 数据库类型
     * @param schema       schema
     * @return SQL
     */
    public static String generateSwitchSchemaSql(Database databaseType, String schema) {
        switch (databaseType) {
            case MYSQL:
                return String.format("USE %s", schema);
            case ORACLE:
                return String.format("ALTER SESSION SET CURRENT_SCHEMA = %s", schema);
            default:
                throw new IllegalStateException("暂不支持数据库：" + databaseType.name());
        }
    }
}
