package com.elitesland.cloudt.tenant.filter;

import com.elitesland.cloudt.tenant.TenantClient;
import com.elitesland.cloudt.tenant.config.support.TenantContextHolder;
import com.elitesland.cloudt.tenant.util.TenantRequestUtil;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.*;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * 租户上下文信息的传递.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/6
 */
@Log4j2
@Activate(group = {"provider", "consumer"})
public class DubboTenantContextFilter implements Filter {

    private static final String ATTACHMENT_KEY = "cloudt_tenant_id";

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        RpcContext rpcContext = RpcContext.getContext();

        if (rpcContext.isProviderSide()) {
            // 服务端负责放置租户信息
            SysTenantDTO tenantDTO = null;
            String tenantId = rpcContext.getAttachment(ATTACHMENT_KEY);
            if (tenantId != null && tenantId.length() > 0) {
                tenantDTO = TenantClient.getTenant(Long.parseLong(tenantId));
                if (tenantDTO != null) {
                    setTenant(tenantDTO);
                    log.info("当前租户：{}", tenantDTO.getTenantCode());
                }
            }

            try {
                return invoker.invoke(invocation);
            } finally {
                if (tenantDTO != null) {
                    clearTenant();
                }
            }
        } else {
            SysTenantDTO tenantDTO = obtainTenant();
            if (tenantDTO != null) {
                rpcContext.setAttachment(ATTACHMENT_KEY, tenantDTO.getId().toString());
            }
            return invoker.invoke(invocation);
        }
    }

    private SysTenantDTO obtainTenant() {
        // 有限从租户上下文中获取
        SysTenantDTO tenantDTO = TenantContextHolder.getCurrentTenant();
        if (tenantDTO != null) {
            return tenantDTO;
        }

        // 从请求信息中获取
        var requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes != null) {
            return TenantRequestUtil.obtainTenant(((ServletRequestAttributes) requestAttributes).getRequest());
        }

        return null;
    }

    private void setTenant(SysTenantDTO tenant) {
        TenantContextHolder.setCurrentTenant(tenant);
    }

    private void clearTenant() {
        TenantContextHolder.clearCurrent();
    }
}
