package com.elitesland.cloudt.tenant.provider;

import com.elitesland.cloudt.context.flyway.FlywayBuilder;
import com.elitesland.cloudt.context.flyway.support.FlywayHelper;
import com.elitesland.cloudt.tenant.config.TenantClientProperties;
import com.elitesland.cloudt.tenant.config.datasource.AbstractTenantDatasourceProvider;
import com.elitesland.yst.common.constant.TenantIsolateStrategy;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;

import java.util.concurrent.TimeUnit;

/**
 * 客户端同步租户信息.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/24
 */
@Log4j2
public class ClientSyncTenantProvider {

    private final TenantClientProperties clientProperties;
    private final TenantProvider tenantProvider;
    private final FlywayHelper flywayHelper;
    private final FlywayBuilder flywayBuilder;

    public ClientSyncTenantProvider(TenantClientProperties clientProperties, TenantProvider tenantProvider,
                                    FlywayHelper flywayHelper, FlywayBuilder flywayBuilder) {
        this.clientProperties = clientProperties;
        this.tenantProvider = tenantProvider;
        this.flywayHelper = flywayHelper;
        this.flywayBuilder = flywayBuilder;
    }

    /**
     * 开始同步
     */
    public boolean sync() {
        try {
            executeSync();
            return true;
        } catch (Exception e) {
            log.error("同步客户端租户数据表结构异常：", e);
        }

        int times = 1;
        while (times < clientProperties.getSyncDbRetryTimes()) {
            try {
                TimeUnit.SECONDS.sleep(clientProperties.getRetryInterval().getSeconds());

                log.info("重试同步客户端租户数据表结构第{}次...", times);
                executeSync();
                log.info("重试同步客户端租户数据表结构第{}次成功！", times);

                return true;
            } catch (Exception e) {
                log.error("第" + times + "次同步客户端租户数据表结构异常：", e);
            }
        }

        return false;
    }

    private void executeSync() {
        var tenants = tenantProvider.getAllTenant().values();
        if (tenants.isEmpty()) {
            log.warn("暂无租户信息");
            return;
        }

        for (SysTenantDTO tenant : tenants) {
            if (tenant.getIsolationStrategy() == null || tenant.getIsolationStrategy() != TenantIsolateStrategy.SCHEMA) {
                log.info("租户【{}, {}】未启用数据独立schema", tenant.getTenantName(), tenant.getId());
                continue;
            }

            // 开始同步
            flywayHelper.migrate(() -> flywayBuilder.createFlywayForTenant(AbstractTenantDatasourceProvider.getDefaultSchema(), tenant.getSchemaName()));
        }
    }
}
