package com.elitesland.cloudt.tenant.transaction;

import com.elitesland.cloudt.context.transaction.TransactionWrapper;
import com.elitesland.cloudt.tenant.TenantClient;
import com.elitesland.cloudt.tenant.config.datasource.TenantSession;
import com.elitesland.yst.security.common.InnerUserEnum;
import com.elitesland.yst.security.entity.GeneralUserDetails;
import com.elitesland.yst.system.dto.SysTenantDTO;
import com.elitesland.yst.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;

import java.util.concurrent.Callable;

/**
 * 租户的事务处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/4/7
 */
@Log4j2
public class TenantTransactionWrapper implements TransactionWrapper<Object, Object> {

    private final TransactionExecutor executor;

    public TenantTransactionWrapper(TransactionExecutor executor) {
        this.executor = executor;
    }

    @Override
    public Object apply(Callable<Object> runnable, Object subject) {
        SysTenantDTO tenant = null;
        if (subject instanceof SysTenantDTO) {
            tenant = (SysTenantDTO) subject;
        } else if (subject instanceof GeneralUserDetails) {
            GeneralUserDetails user = (GeneralUserDetails) subject;
            tenant = InnerUserEnum.ADMIN.getUsername().equals(user.getUsername()) ? null : user.getTenant();
        } else if (subject instanceof SysUserDTO) {
            SysUserDTO user = (SysUserDTO) subject;
            if (!InnerUserEnum.ADMIN.getUsername().equals(user.getUsername())) {
                tenant = user.getSysTenantVO();
            }
        } else if (subject instanceof Long) {
            tenant = TenantClient.getTenant((Long) subject);
        } else if (subject != null) {
            throw new IllegalArgumentException("未知参数类型");
        }

        if (tenant == null) {
            // 租户为空，则使用默认schema
            return applyDefaultSchema(runnable);
        }

        // 使用对应租户的schema
        return applyTenantSchema(runnable, tenant);
    }

    private Object applyDefaultSchema(Callable<Object> runnable) {
        TenantSession.setUseDefaultSchema();

        try {
            return executor.execute(runnable);
        } catch (Exception e) {
            throw new RuntimeException("【租户】事务包装处理异常", e);
        } finally {
            TenantSession.clearUseDefaultSchema();
        }
    }

    private Object applyTenantSchema(Callable<Object> runnable, SysTenantDTO tenant) {
        // 如果之前使用默认的，则取消使用默认
        var useDefaultSchema = TenantSession.getUseDefaultSchema();
        if (useDefaultSchema) {
            TenantSession.clearUseDefaultSchema();
        }

        var originalTenant = TenantSession.getCurrentTenant();

        Assert.notNull(tenant, "租户不存在");
        TenantSession.setCurrentTenant(tenant);

        try {
            return executor.execute(runnable);
        } catch (Exception e) {
            throw new RuntimeException("【租户】事务包装处理异常", e);
        } finally {
            if (originalTenant == null) {
                TenantSession.clearCurrent();
            } else {
                TenantSession.setCurrentTenant(originalTenant);
            }

            if (useDefaultSchema) {
                TenantSession.setUseDefaultSchema();
            }
        }
    }
}
