package com.elitesland.cbpl.cloudt.tenant;

import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitesland.cbpl.cloudt.iam.IamUserUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.skywalking.apm.toolkit.trace.RunnableWrapper;
import org.apache.skywalking.apm.toolkit.trace.SupplierWrapper;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

/**
 * @author eric.hao
 * @since 2023/07/10
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class TenantWrapper {

    /**
     * 使用CloudTi架构提供的线程池
     * {@link com.elitescloud.boot.threadpool.ThreadPoolAutoConfiguration}
     */
    @Resource
    private TaskExecutor taskExecutor;

    /**
     * 租户隔离工具类
     */
    @Resource
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    /**
     * 异步线程封装：会根据Session切换当前租户
     * <p>
     * 以Supplier函数式接口类型为参数，返回结果类型为T；Supplier接口的 get()是有返回值的(会阻塞)
     *
     * @param executor 业务侧执行方法
     * @return 异步对象
     */
    public <T> CompletableFuture<T> supplyAsync(Supplier<T> executor) {
        log.debug("[PHOENIX-CLOUDT] CURRENT TENANT: {}", TenantClient.getSessionTenant());
        return CompletableFuture.supplyAsync(SupplierWrapper.of(() -> byTenantDirectly(executor)), taskExecutor);
    }

    /**
     * 异步线程封装：会根据Session切换当前租户
     * <p>
     * 以Runnable函数式接口类型为参数，没有返回结果
     *
     * @param executor 业务侧执行方法
     * @return 异步对象
     */
    public <T> CompletableFuture<Void> runAsync(Runnable executor) {
        log.debug("[PHOENIX-CLOUDT] CURRENT TENANT: {}", TenantClient.getSessionTenant());
        return CompletableFuture.runAsync(RunnableWrapper.of(() -> byTenantDirectly(executor)), taskExecutor);
    }

    /**
     * 直接使用指定的租户
     *
     * @param executor   业务操作
     * @param tenantCode 租户编码
     * @param <T>        数据类型
     * @return 数据
     */
    public <T> T byTenantDirectly(Supplier<T> executor, String tenantCode) {
        return tenantDataIsolateProvider.byTenantDirectly(executor, tenantCode);
    }

    public <T> T byTenantDirectly(Supplier<T> executor) {
        return byTenantDirectly(executor, IamUserUtil.currentTenantCode());
    }

    /**
     * 直接使用指定的租户
     *
     * @param executor   业务操作
     * @param tenantCode 租户编码
     */
    public void byTenantDirectly(Runnable executor, String tenantCode) {
        Supplier<Void> useSupplier = () -> {
            executor.run();
            return null;
        };
        tenantDataIsolateProvider.byTenantDirectly(useSupplier, tenantCode);
    }

    public void byTenantDirectly(Runnable executor) {
        byTenantDirectly(executor, IamUserUtil.currentTenantCode());
    }
}
