package com.elitesland.cbpl.cloudt.iam;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.cbpl.tool.core.http.HttpServletUtil;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2022/03/22
 */
@Slf4j
public class IamUserUtil {

    private static final Long DEFAULT_TENANT_ID = -1L;
    private static final String CURR_USER_ERROR = "获取当前用户信息失败";

    /**
     * 是否登录
     */
    public static boolean isLogin() {
        var user = SecurityContextUtil.currentUser();
        return ObjectUtil.isNull(user);
    }

    /**
     * 当前登录人 - 用户信息
     */
    public static GeneralUserDetails currentUser() {
        var user = SecurityContextUtil.currentUser();
        if (ObjectUtil.isNull(user)) {
            throw new BusinessException(CURR_USER_ERROR);
        }
        return user;
    }


    /**
     * 当前登录人 - 用户名
     */
    public static String currentUserName() {
        String userName = SecurityContextUtil.currentUserName();
        if (StrUtil.isBlank(userName)) {
            throw new BusinessException(CURR_USER_ERROR);
        }
        return userName;
    }

    /**
     * 当前登录人 - userId
     */
    public static Long currentUserId() {
        Long userId = SecurityContextUtil.currentUserId();
        if (ObjectUtil.isNull(userId)) {
            throw new BusinessException(CURR_USER_ERROR);
        }
        return userId;
    }

    /**
     * 当前登录人 - userId
     *
     * @return 字符串类型
     */
    public static String currentUserIdStr() {
        return String.valueOf(currentUserId());
    }

    /**
     * 当前登录人 - 租户ID
     */
    public static Long currentTenantId() {
        SysTenantDTO tenant = currentTenant();
        return ObjectUtil.isNull(tenant.getId()) ? DEFAULT_TENANT_ID : tenant.getId();
    }

    /**
     * 当前登录人 - 租户ID
     *
     * @return 字符串类型
     */
    public static String currentTenantIdStr() {
        return String.valueOf(currentTenantId());
    }

    /**
     * 当前登录人 - 所属ouId
     */
    public static Long currentUserOuId() {
        var user = SecurityContextUtil.currentUser();
        if (ObjectUtil.isNull(user)) {
            throw new BusinessException(CURR_USER_ERROR);
        }
        return Optional.of(user).map(GeneralUserDetails::getUser).map(SysUserDTO::getOuId).orElse(0L);
    }

    /**
     * 当前登录人 - 租户信息
     */
    public static SysTenantDTO currentTenant() {
        return TenantClient.getSessionTenant();
    }

    /**
     * 当前登录人 - 从cookie中获取租户id
     */
    public static long currentTenantIdByCookie() {
        HttpServletRequest request = HttpServletUtil.currentRequest();
        Cookie[] cookies = request.getCookies();
        if (ObjectUtil.isNotNull(cookies)) {
            List<String> tokens = Arrays.stream(cookies).filter(cookie -> "token".equals(cookie.getName())).map(Cookie::getValue).collect(Collectors.toList());
            var user = CollUtil.isNotEmpty(tokens) ? SecurityContextUtil.convertToken(tokens.get(0).replace("Bearer%20", "")) : null;
            return Optional.ofNullable(user).map(GeneralUserDetails::getUser).map(SysUserDTO::getTenantId).orElse(DEFAULT_TENANT_ID);
        }
        return DEFAULT_TENANT_ID;
    }

    /**
     * 获取当前登录人的token值
     *
     * @return 当前的token值
     */
    public static String currentToken() {
        return SecurityContextUtil.currentToken();
    }
}