package com.elitesland.cbpl.tool.api.domain;

import cn.hutool.http.HttpStatus;
import com.elitesland.cbpl.common.base.ResponseVO;
import lombok.Data;

import javax.annotation.Nullable;
import java.util.Date;

/**
 * @author eric.hao
 * @since 2024/01/17
 */
@Data
public class HttpResult<T> extends ResponseVO {

    /**
     * 错误号
     */
    @Nullable
    private String errorNo;

    /**
     * 响应数据
     */
    @Nullable
    private T data;

    public static <T> HttpResult<T> ok() {
        return ok(null);
    }

    public static <T> HttpResult<T> ok(T data) {
        HttpResult<T> result = result(HttpStatus.HTTP_OK, "");
        result.setData(data);
        return result;
    }

    public static <T> HttpResult<T> error() {
        return error(null);
    }

    public static <T> HttpResult<T> error(String msg) {
        return error(null, msg);
    }

    public static <T> HttpResult<T> error(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_INTERNAL_ERROR, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    // ========= other

    /**
     * @since 0.1.11-SNAPSHOT
     */
    public static <T> HttpResult<T> badRequest() {
        return badRequest(null, "400: Bad Request.");
    }

    public static <T> HttpResult<T> badRequest(String msg) {
        return badRequest(null, msg);
    }

    public static <T> HttpResult<T> badRequest(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_BAD_REQUEST, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> unauthorized() {
        return unauthorized(null, "401: Unauthorized.");
    }

    public static <T> HttpResult<T> unauthorized(String msg) {
        return unauthorized(null, msg);
    }

    public static <T> HttpResult<T> unauthorized(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_UNAUTHORIZED, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> notFound() {
        return notFound(null, "404: Not Found.");
    }

    public static <T> HttpResult<T> notFound(String msg) {
        return notFound(null, msg);
    }

    public static <T> HttpResult<T> notFound(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_NOT_FOUND, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> requestEntityTooLarge() {
        return requestEntityTooLarge(null, "413: Request Entity Too Large.");
    }

    public static <T> HttpResult<T> requestEntityTooLarge(String msg) {
        return requestEntityTooLarge(null, msg);
    }

    public static <T> HttpResult<T> requestEntityTooLarge(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_ENTITY_TOO_LARGE, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> internalServerError() {
        return internalServerError(null, "500: Internal Server Error.");
    }

    public static <T> HttpResult<T> internalServerError(String msg) {
        return internalServerError(null, msg);
    }

    public static <T> HttpResult<T> internalServerError(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_INTERNAL_ERROR, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> notImplemented() {
        return notImplemented(null, "501: Not Implemented.");
    }

    public static <T> HttpResult<T> notImplemented(String msg) {
        return notImplemented(null, msg);
    }

    public static <T> HttpResult<T> notImplemented(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_NOT_IMPLEMENTED, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> bagGateway() {
        return bagGateway(null, "502: Bad Gateway.");
    }

    public static <T> HttpResult<T> bagGateway(String msg) {
        return bagGateway(null, msg);
    }

    public static <T> HttpResult<T> bagGateway(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_BAD_GATEWAY, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> serviceUnavailable() {
        return serviceUnavailable(null, "503: Service Unavailable.");
    }

    public static <T> HttpResult<T> serviceUnavailable(String msg) {
        return serviceUnavailable(null, msg);
    }

    public static <T> HttpResult<T> serviceUnavailable(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_UNAVAILABLE, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> gatewayTimeout() {
        return gatewayTimeout(null, "504: Gateway Timeout.");
    }

    public static <T> HttpResult<T> gatewayTimeout(String msg) {
        return gatewayTimeout(null, msg);
    }

    public static <T> HttpResult<T> gatewayTimeout(String errorNo, String msg) {
        HttpResult<T> result = result(HttpStatus.HTTP_GATEWAY_TIMEOUT, msg);
        result.setErrorNo(errorNo);
        return result;
    }

    public static <T> HttpResult<T> result(int code, String msg) {
        return result(code, null, msg);
    }

    public static <T> HttpResult<T> result(int code, String errorNo, String msg) {
        HttpResult<T> result = new HttpResult<>();
        result.setCode(code);
        result.setMsg(msg);
        result.setErrorNo(errorNo);
        result.setTime(new Date());
        return result;
    }
}
