package com.elitesland.cbpl.tool.core.bean;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.json.JSON;
import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.exc.InvalidDefinitionException;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;

/**
 * Bean工具类
 *
 * @author eric.hao
 * @since 2023/06/11
 */
@Slf4j
public class BeanUtils extends BeanUtil {

    private static final ObjectMapper OBJECT_MAPPER;

    static {
        OBJECT_MAPPER = new ObjectMapper();
        OBJECT_MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    /**
     * JsonStr => Bean
     */
    @SneakyThrows
    public static <T> T toBean(String source, TypeReference<T> typeRef) {
        Objects.requireNonNull(source);
        Objects.requireNonNull(typeRef);
        return OBJECT_MAPPER.readValue(source, typeRef);
    }

    /**
     * JsonStr => Bean
     */
    @SneakyThrows
    public static <T> T toBean(String source, Class<T> clazz) {
        Objects.requireNonNull(source);
        Objects.requireNonNull(clazz);
        return OBJECT_MAPPER.readValue(source, clazz);
    }

    /**
     * Map => Bean
     */
    public static <T> T toBean(Map<String, ?> source, Class<T> clazz) {
        return toBean(toJsonStr(source), clazz);
    }

    /**
     * Map => Bean
     */
    public static <T> T toBean(Map<String, ?> source, TypeReference<T> typeRef) {
        return toBean(toJsonStr(source), typeRef);
    }

    /**
     * JsonStr => Map
     */
    @SneakyThrows
    public static Map<String, Object> toMap(String source) {
        return toMap(source, Object.class);
    }

    /**
     * JsonStr => Map
     *
     * @since 0.1.12-SNAPSHOT
     */
    @SneakyThrows
    public static <T> Map<String, T> toMap(String source, Class<T> clazz) {
        return toMap(source, new TypeReference<>() {
        });
    }

    /**
     * JsonStr => Map
     *
     * @since 0.1.12-SNAPSHOT
     */
    @SneakyThrows
    public static <T> T toMap(String source, TypeReference<T> typeRef) {
        Objects.requireNonNull(source);
        return OBJECT_MAPPER.readValue(source, typeRef);
    }

    /**
     * Bean => Map
     */
    public static Map<String, Object> beanToMap(Object bean, String... properties) {
        return BeanUtil.beanToMap(bean, properties);
    }

    /**
     * JsonStr => BeanList
     */
    public static <T> List<T> toList(String source, Class<T> clazz) {
        return toList(toMapList(source), clazz);
    }

    /**
     * MapList => BeanList
     */
    public static <T> List<T> toList(List<Map<String, Object>> source, Class<T> clazz) {
        return source.stream().map(row -> toBean(row, clazz)).filter(Objects::nonNull).collect(Collectors.toList());
    }

    /**
     * Object[] => BeanList
     */
    @SneakyThrows
    public static <T> List<T> toList(Object[] source, Class<T> clazz) {
        Objects.requireNonNull(source);
        Objects.requireNonNull(clazz);
        return OBJECT_MAPPER.readValue(toJsonStr(source), new TypeReference<>() {
        });
    }

    /**
     * Object[] => StrList
     */
    public static List<String> toStrList(Object[] source) {
        return toList(source, String.class);
    }

    /**
     * Object[] => IntegerList
     */
    public static List<Integer> toIntegerList(Object[] source) {
        return toList(source, Integer.class);
    }

    /**
     * JsonStr => MapList
     */
    @SneakyThrows
    public static List<Map<String, Object>> toMapList(String source) {
        Objects.requireNonNull(source);
        return OBJECT_MAPPER.readValue(source, new TypeReference<>() {
        });
    }

    /**
     * BeanList => MapList
     */
    @SneakyThrows
    public static List<Map<String, Object>> toMapList(List<?> source) {
        Objects.requireNonNull(source);
        return OBJECT_MAPPER.readValue(toJsonStr(source), new TypeReference<>() {
        });
    }

    /**
     * Object => toJsonOrThrow
     */
    public static <T> String toJsonStr(T payload) {
        return toJsonOrThrow(payload);
    }

    /**
     * Object => Optional[JsonStr]
     */
    public static Optional<String> toJsonOptional(Object obj) {
        if (obj == null) return Optional.empty();
        try {
            return Optional.of(OBJECT_MAPPER.writeValueAsString(obj));
        } catch (InvalidDefinitionException e) {
            JSON parse = JSONUtil.parse(obj);
            return Optional.of(parse.toStringPretty());
        } catch (JsonProcessingException e) {
            logger.warn("[PHOENIX-JSON] Convert object to json FAILED: {}", e.getMessage());
            return Optional.empty();
        }
    }

    /**
     * Object => JsonStr
     * <li>解析错误时，返回空字符串</li>
     */
    public static String toJsonOrEmpty(Object obj) {
        return toJsonOptional(obj).orElse("");
    }

    /**
     * Object => JsonStr
     * <li>解析错误是，抛出异常</li>
     */
    public static String toJsonOrThrow(Object obj) {
        return toJsonOptional(obj).orElseThrow(() -> new IllegalArgumentException(
                "[PHOENIX-JSON] Convert object to json FAILED: " + obj));
    }
}
