package com.elitesland.cbpl.tool.core.util;

import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.reflections.scanners.Scanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.springframework.util.ReflectionUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.*;
import java.util.stream.Collectors;

import static org.reflections.scanners.Scanners.SubTypes;
import static org.reflections.scanners.Scanners.TypesAnnotated;

/**
 * @author eric.hao
 * @since 2024/02/24
 */
@Slf4j
public class ReflectUtils {

    /**
     * 反射类扫描注册
     */
    public static Reflections reflections(List<String> basePackages) {
        return reflections(basePackages, TypesAnnotated, SubTypes);
    }

    public static Reflections reflections(List<String> basePackages, Scanner... scanners) {
        Collection<URL> urls = new ArrayList<>();
        for (String basePackage : basePackages) {
            urls.addAll(ClasspathHelper.forPackage(basePackage));
        }
        return new Reflections(new ConfigurationBuilder().setUrls(urls).addScanners(scanners));
    }

    /**
     * 从 Spring 容器中，获取注解相关的类
     *
     * @param annotation 指定注解
     * @return 类列表
     */
    public static Set<Class<?>> getTypesAnnotatedWith(Class<? extends Annotation> annotation) {
        Reflections reflections = SpringUtils.getBean(Reflections.class);
        if (ObjectUtil.isEmpty(reflections)) {
            logger.debug("Reflections Not Declaring.");
            return Collections.emptySet();
        }
        return reflections.getTypesAnnotatedWith(annotation);
    }

    /**
     * 获取字段值
     *
     * @param instance  对象，如果static字段，此处为类
     * @param fieldName 字段名
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    public static Object getFieldValue(Object instance, String fieldName) {
        Object fieldValue;
        if (instance instanceof Map) {
            fieldValue = ((Map<?, ?>) instance).get(fieldName);
        } else {
            fieldValue = ReflectUtil.getFieldValue(instance, fieldName);
        }
        return fieldValue;
    }

    /**
     * 获取字段值，默认返回空串
     *
     * @param instance  对象，如果static字段，此处为类
     * @param fieldName 字段名
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    public static String getFieldValueOrDefault(Object instance, String fieldName) {
        return getFieldValueOrDefault(instance, fieldName, "");
    }

    /**
     * 获取字段值
     *
     * @param instance     对象，如果static字段，此处为类
     * @param fieldName    字段名
     * @param defaultValue 默认值
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    @SuppressWarnings("unchecked")
    public static <T> T getFieldValueOrDefault(Object instance, String fieldName, T defaultValue) {
        Object fieldValue = getFieldValue(instance, fieldName);
        return ObjectUtil.isEmpty(fieldValue) ? defaultValue : (T) fieldValue;
    }

    /**
     * 指定类的方法调用
     * <blockquote>
     * 参考：{@code Class t = Class.forName("java.lang.Thread")}
     * </blockquote>
     *
     * @param className  类名
     * @param methodName 方法名
     * @return 方法执行结果
     */
    public static Object invokeMethod(String className, String methodName) {
        Object bean = SpringUtils.getClassBean(className);
        return invokeMethod(bean, methodName);
    }

    /**
     * 指定类的方法调用
     *
     * @param instance   类实例
     * @param methodName 方法名
     * @return 方法执行结果
     */
    public static Object invokeMethod(Object instance, String methodName) {
        Method method = ReflectionUtils.findMethod(instance.getClass(), methodName);
        return ReflectionUtils.invokeMethod(method, instance);
    }

    /**
     * 获取指定类的所有子类/实现类
     *
     * @since 0.1.11-SNAPSHOT
     */
    public static <T> List<T> getSubTypesOf(Class<T> superClass) {
        Reflections reflections = SpringUtils.getBean(Reflections.class);
        if (ObjectUtil.isEmpty(reflections)) {
            logger.debug("Reflections Not Declaring.");
            return Collections.emptyList();
        }
        return reflections.getSubTypesOf(superClass).stream().map(SpringUtils::getBean).collect(Collectors.toList());
    }
}
