package com.elitesland.cbpl.tool.core.util;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

/**
 * 参考 云梯锁工具类
 *
 * @author eric.hao
 * @version 0.1.15-SNAPSHOT
 * @since 2024/08/30
 */
@Slf4j
public class LockUtil {

    private static final ConcurrentHashMap<String, Lock> LOCAL_LOCK = new ConcurrentHashMap<>(512);
    private static RedissonClient redissonClient;

    private LockUtil() {
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier) {
        return executeByLock(lockKey, supplier, null);
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration) {
        return executeByLock(lockKey, supplier, duration, null);
    }

    /**
     * 加锁执行，定时任务场景
     * <li>有锁直接抛出异常</li>
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T schedulerByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration) {
        return executeByLock(lockKey, supplier, duration, null, false);
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration, String acquireFailMsg) {
        return executeByLock(lockKey, supplier, duration, acquireFailMsg, true);
    }

    /**
     * 加锁执行
     *
     * @param lockKey      锁的标识，需唯一
     * @param supplier     需要加锁的方法体
     * @param duration     最多等待获取锁的时间
     * @param waitStrategy true等待获取锁；false直接抛出异常；
     * @param <T>          结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration, String acquireFailMsg, boolean waitStrategy) {
        Assert.hasText(lockKey, "锁的标识为空");
        if (duration == null) {
            duration = Duration.ofMinutes(10);
        }

        // 优先获取本地锁
        var lockLocal = LOCAL_LOCK.computeIfAbsent(lockKey, c -> new ReentrantLock());
        RLock lockRemote = null;
        try {
            logger.info("尝试获取本地锁：{}，{}", lockKey, duration);
            if (lockLocal.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                // 尝试获取远程锁
                lockRemote = getRedissonClient().getLock(lockKey);
                logger.info("尝试获取远程锁：{}，{}", lockKey, duration);
                // 定时任务场景 && 锁存在
                if (!waitStrategy && lockRemote.isLocked()) {
                    throw PhoenixException.unexpected("已有任务正在进行中");
                }
                if (lockRemote.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                    return supplier.get();
                }
            }

            throw PhoenixException.unexpected(CharSequenceUtil.blankToDefault(acquireFailMsg, "未获取到锁" + lockKey + "，等待超时"));
        } catch (Throwable e) {
            logger.error("加锁的方法中出现异常：{}", lockKey, e);
            if (e instanceof RuntimeException) {
                throw (PhoenixException) e;
            }
            throw new RuntimeException(e);
        } finally {
            // 如果是定时任务场景，则额外增加锁持有时间
            if (!waitStrategy) {
                try {
                    Thread.sleep(duration.toMillis());
                } catch (InterruptedException e) {
                    logger.error("InterruptedException：{}", lockKey, e);
                }
            }
            // 释放锁
            logger.info("释放本地锁：{}", lockKey);
            try {
                lockLocal.unlock();
            } catch (Exception e) {
                logger.error("本地锁释放异常：{}", lockKey, e);
            }
            if (lockRemote != null && lockRemote.isLocked()) {
                logger.info("释放远程锁：{}", lockKey);
                try {
                    lockRemote.unlock();
                } catch (Exception e) {
                    logger.error("远程锁释放异常：{}", lockKey, e);
                }
            }
        }
    }

    private static RedissonClient getRedissonClient() {
        if (redissonClient == null) {
            redissonClient = SpringUtils.getBean(RedissonClient.class);
        }
        return redissonClient;
    }
}
