package com.elitesland.cbpl.tool.websocket.config;

import com.corundumstudio.socketio.Configuration;
import com.corundumstudio.socketio.SocketConfig;
import com.corundumstudio.socketio.SocketIOServer;
import com.corundumstudio.socketio.annotation.SpringAnnotationScanner;
import com.elitesland.cbpl.tool.websocket.registrar.WebSocketListener;
import com.elitesland.cbpl.tool.websocket.registrar.WebSocketInitialize;
import com.elitesland.cbpl.tool.websocket.notifier.WebSocketNotifier;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.web.socket.config.annotation.EnableWebSocket;

import static com.elitesland.cbpl.tool.websocket.config.WebSocketProperties.WX_CONFIG_PREFIX;

/**
 * WebSocket 相关配置
 *
 * @author eric.hao
 * @since 2022/09/14
 */
@Slf4j
@EnableWebSocket
@org.springframework.context.annotation.Configuration
@EnableConfigurationProperties(WebSocketProperties.class)
@ConditionalOnProperty(prefix = WX_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class WebSocketConfig {

    private final WebSocketProperties webSocketProperties;

    /**
     * 注册Websocket服务端地址
     *
     * @since 0.1.16-SNAPSHOT
     */
    @Bean
    public SocketIOServer socketIOServer() {
        SocketConfig socketConfig = new SocketConfig();
        socketConfig.setSoLinger(0);
        socketConfig.setReuseAddress(true);

        Configuration config = new Configuration();
        config.setSocketConfig(socketConfig);
        // TODO 考虑多节点集群部署，暂不通过配置，设置固定host
        // config.setHostname(webSocketProperties.getHost());
        config.setPort(webSocketProperties.getPort());
        config.setContext(webSocketProperties.getContextPath());

        config.setBossThreads(webSocketProperties.getBossCount());
        config.setWorkerThreads(webSocketProperties.getBossCount());
        config.setAllowCustomRequests(webSocketProperties.isAllowCustomRequests());
        config.setUpgradeTimeout(webSocketProperties.getUpgradeTimeout());
        config.setPingTimeout(webSocketProperties.getPingTimeout());
        config.setPingInterval(webSocketProperties.getPingInterval());

        // 设置 sessionId 随机
        config.setRandomSession(true);
        config.setOrigin("*");

        SocketIOServer server = new SocketIOServer(config);
        server.addNamespace(webSocketProperties.getNamespace());
        var url = webSocketProperties.getHost() + ":" + webSocketProperties.getPort() + webSocketProperties.getNamespace();
        logger.info("[PHOENIX-WS] websocket server namespace: {}", url);
        return server;
    }

    /**
     * Websocket 通知类
     */
    @Bean
    @ConditionalOnMissingBean
    public WebSocketNotifier webSocketPusher(SocketIOServer socketServer) {
        logger.info("[PHOENIX-WS] websocket message pusher");
        return new WebSocketNotifier(socketServer, webSocketProperties);
    }

    @Bean
    public SpringAnnotationScanner springAnnotationScanner(SocketIOServer socketServer) {
        return new SpringAnnotationScanner(socketServer);
    }

    /**
     * 启动添加监听(连接、断开)
     */
    @Bean
    public WebSocketListener webSocketListener(SocketIOServer socketServer) {
        return new WebSocketListener(socketServer);
    }

    /**
     * 启动注册
     */
    @Bean
    public WebSocketInitialize webSocketInitialize(SocketIOServer socketIOServer) {
        return new WebSocketInitialize(socketIOServer, webSocketProperties);
    }
}
