package com.elitesland.cbpl.tool.websocket.config;

import com.elitesland.cbpl.tool.websocket.handler.WebSocketAuthHandler;
import com.elitesland.cbpl.tool.websocket.interceptor.CustomWebSocketInterceptor;
import com.elitesland.cbpl.tool.websocket.notifier.WebSocketNotifier;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.socket.config.annotation.EnableWebSocket;
import org.springframework.web.socket.config.annotation.WebSocketConfigurer;
import org.springframework.web.socket.config.annotation.WebSocketHandlerRegistry;

import static com.elitesland.cbpl.tool.websocket.config.WebSocketProperties.WX_CONFIG_PREFIX;

/**
 * WebSocket 相关配置
 *
 * @author eric.hao
 * @since 2022/09/14
 */
@Slf4j
@EnableWebSocket
@Configuration
@EnableConfigurationProperties(WebSocketProperties.class)
@ConditionalOnProperty(prefix = WX_CONFIG_PREFIX, name = "enabled", havingValue = "true")
@RequiredArgsConstructor
public class WebSocketConfig implements WebSocketConfigurer {

    private final WebSocketProperties webSocketProperties;

    /**
     * Websocket 通知类
     */
    @Bean
    @ConditionalOnMissingBean
    public WebSocketNotifier webSocketPusher() {
        logger.info("[PHOENIX-WS] websocket message pusher");
        return new WebSocketNotifier();
    }

    /**
     * 注册握手请求拦截器
     */
    @Bean
    public CustomWebSocketInterceptor customWebSocketInterceptor() {
        return new CustomWebSocketInterceptor();
    }

    /**
     * 注册Websocket服务端地址
     */
    @Override
    public void registerWebSocketHandlers(WebSocketHandlerRegistry registry) {
        logger.info("[PHOENIX-WS] websocket message handler: {}", webSocketProperties.getEntry());
        registry.addHandler(new WebSocketAuthHandler(), webSocketProperties.getEntry())
                .setAllowedOrigins(webSocketProperties.getAllow())
                .withSockJS();
    }
}
