package com.elitesland.cbpl.tool.core.util;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.cbpl.tool.core.exceptions.PhoenixException;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.util.Assert;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.Duration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

/**
 * 参考 云梯锁工具类
 *
 * @author eric.hao
 * @version 0.1.15-SNAPSHOT
 * @since 2024/08/30
 */
@Slf4j
public class LockUtil {

    private static final ConcurrentHashMap<String, Lock> LOCAL_LOCK = new ConcurrentHashMap<>(512);
    private static RedissonClient redissonClient;

    private LockUtil() {
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier) {
        return executeByLock(lockKey, supplier, null);
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration) {
        return executeByLock(lockKey, supplier, duration, null);
    }

    /**
     * 加锁执行，定时任务场景
     *
     * @param taskCode 任务编码
     * @param executor 需要加锁的方法体
     * @param <T>      结果类型
     * @since 0.1.17-SNAPSHOT
     */
    public static <T> void schedulerByLock(@NotBlank String taskCode, @NotNull Runnable executor) {
        // 不等待获取锁时间
        schedulerByLock(taskCode, executor, 0L, TimeUnit.SECONDS);
    }

    /**
     * 加锁执行，定时任务场景
     *
     * @param taskCode 任务编码
     * @param executor 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param time     单位
     * @since 0.1.16-SNAPSHOT
     * @deprecated The method {@link #schedulerByLock(String, Runnable)}
     */
    @Deprecated(since = "0.1.17-SNAPSHOT", forRemoval = true)
    public static void schedulerByLock(@NotBlank String taskCode, @NotNull Runnable executor, final long duration, TimeUnit time) {
        Assert.hasText(taskCode, "锁的标识为空");

        // 优先获取本地锁
        var lockLocal = LOCAL_LOCK.computeIfAbsent(taskCode, c -> new ReentrantLock());
        RLock lockRemote = null;
        try {
            logger.info("尝试获取本地锁：{}", taskCode);
            if (lockLocal.tryLock()) {
                // 尝试获取远程锁
                lockRemote = getRedissonClient().getLock(taskCode);
                logger.info("尝试获取远程锁：{}", taskCode);
                if (lockRemote.tryLock()) {
                    executor.run();
                } else {
                    logger.warn("远程锁获取失败-{}：已有任务正在进行中", taskCode);
                }
            } else {
                logger.warn("本地锁获取失败-{}：已有任务正在进行中", taskCode);
            }
        } catch (Throwable e) {
            logger.error("加锁的方法中出现异常：{}", taskCode, e);
            throw new RuntimeException(e);
        } finally {
            // 释放锁
            logger.info("释放本地锁：{}", taskCode);
            try {
                lockLocal.unlock();
            } catch (Exception e) {
                logger.error("本地锁释放异常：{}", taskCode, e);
            }
            if (lockRemote != null && lockRemote.isLocked()) {
                logger.info("释放远程锁：{}", taskCode);
                try {
                    lockRemote.unlock();
                } catch (Exception e) {
                    logger.error("远程锁释放异常：{}", taskCode, e);
                }
            }
        }
    }

    /**
     * 加锁执行
     *
     * @param lockKey  锁的标识，需唯一
     * @param supplier 需要加锁的方法体
     * @param duration 最多等待获取锁的时间
     * @param <T>      结果类型
     * @return 方法体返回的结果
     */
    public static <T> T executeByLock(@NotBlank String lockKey, @NotNull Supplier<T> supplier, Duration duration, String acquireFailMsg) {
        Assert.hasText(lockKey, "锁的标识为空");
        if (duration == null) {
            duration = Duration.ofMinutes(10);
        }

        // 优先获取本地锁
        var lockLocal = LOCAL_LOCK.computeIfAbsent(lockKey, c -> new ReentrantLock());
        RLock lockRemote = null;
        try {
            logger.info("尝试获取本地锁：{}，{}", lockKey, duration);
            if (lockLocal.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                // 尝试获取远程锁
                lockRemote = getRedissonClient().getLock(lockKey);
                logger.info("尝试获取远程锁：{}，{}", lockKey, duration);
                if (lockRemote.tryLock(duration.getSeconds(), TimeUnit.SECONDS)) {
                    return supplier.get();
                }
            }

            throw PhoenixException.unexpected(CharSequenceUtil.blankToDefault(acquireFailMsg, "未获取到锁" + lockKey + "，等待超时"));
        } catch (Throwable e) {
            logger.error("加锁的方法中出现异常：{}", lockKey, e);
            if (e instanceof RuntimeException) {
                throw (PhoenixException) e;
            }
            throw new RuntimeException(e);
        } finally {
            // 释放锁
            logger.info("释放本地锁：{}", lockKey);
            try {
                lockLocal.unlock();
            } catch (Exception e) {
                logger.error("本地锁释放异常：{}", lockKey, e);
            }
            if (lockRemote != null && lockRemote.isLocked()) {
                logger.info("释放远程锁：{}", lockKey);
                try {
                    lockRemote.unlock();
                } catch (Exception e) {
                    logger.error("远程锁释放异常：{}", lockKey, e);
                }
            }
        }
    }

    private static RedissonClient getRedissonClient() {
        if (redissonClient == null) {
            redissonClient = SpringUtils.getBean(RedissonClient.class);
        }
        return redissonClient;
    }
}
