package com.elitesland.cbpl.tool.core.util;

import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.cbpl.tool.core.bean.BeanUtils;
import com.elitesland.cbpl.tool.extra.spring.SpringUtils;
import lombok.extern.slf4j.Slf4j;
import org.reflections.Reflections;
import org.reflections.scanners.Scanner;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.springframework.lang.Nullable;
import org.springframework.util.ReflectionUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.net.URL;
import java.util.*;
import java.util.stream.Collectors;

import static org.reflections.scanners.Scanners.SubTypes;
import static org.reflections.scanners.Scanners.TypesAnnotated;

/**
 * @author eric.hao
 * @since 2024/02/24
 */
@Slf4j
public class ReflectUtils {

    /**
     * 反射类扫描注册
     */
    public static Reflections reflections(List<String> basePackages) {
        return reflections(basePackages, TypesAnnotated, SubTypes);
    }

    public static Reflections reflections(List<String> basePackages, Scanner... scanners) {
        Collection<URL> urls = new ArrayList<>();
        for (String basePackage : basePackages) {
            urls.addAll(ClasspathHelper.forPackage(basePackage));
        }
        return new Reflections(new ConfigurationBuilder().setUrls(urls).addScanners(scanners));
    }

    /**
     * 从 Spring 容器中，获取注解相关的类
     *
     * @param annotation 指定注解
     * @return 类列表
     */
    public static Set<Class<?>> getTypesAnnotatedWith(Class<? extends Annotation> annotation) {
        Reflections reflections = SpringUtils.getBean(Reflections.class);
        if (ObjectUtil.isEmpty(reflections)) {
            logger.debug("Reflections Not Declaring.");
            return Collections.emptySet();
        }
        return reflections.getTypesAnnotatedWith(annotation);
    }

    /**
     * 获取字段值
     *
     * @param instance  对象，如果static字段，此处为类
     * @param fieldName 字段名
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    public static Object getFieldValue(Object instance, String fieldName) {
        Object fieldValue;
        if (instance instanceof Map) {
            fieldValue = ((Map<?, ?>) instance).get(fieldName);
        } else {
            fieldValue = ReflectUtil.getFieldValue(instance, fieldName);
        }
        return fieldValue;
    }

    /**
     * 获取字段值，默认返回空串
     *
     * @param instance  对象，如果static字段，此处为类
     * @param fieldName 字段名
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    public static String getFieldValueOrDefault(Object instance, String fieldName) {
        return getFieldValueOrDefault(instance, fieldName, "");
    }

    /**
     * 获取字段值
     *
     * @param instance     对象，如果static字段，此处为类
     * @param fieldName    字段名
     * @param defaultValue 默认值
     * @return 字段值
     * @throws UtilException 包装IllegalAccessException异常
     */
    @SuppressWarnings("unchecked")
    public static <T> T getFieldValueOrDefault(Object instance, String fieldName, T defaultValue) {
        Object fieldValue = getFieldValue(instance, fieldName);
        return ObjectUtil.isEmpty(fieldValue) ? defaultValue : (T) fieldValue;
    }

    /**
     * 指定类的方法调用
     * <blockquote>
     * 参考：{@code Class t = Class.forName("java.lang.Thread")}
     * </blockquote>
     *
     * @param className  类名
     * @param methodName 方法名
     * @return 方法执行结果
     */
    public static Object invokeMethod(String className, String methodName) {
        Object bean = SpringUtils.getClassBean(className);
        return invokeMethod(bean, methodName);
    }

    /**
     * 指定类的方法调用
     *
     * @param instance   类实例
     * @param methodName 方法名
     * @return 方法执行结果
     */
    public static Object invokeMethod(Object instance, String methodName) {
        Method method = ReflectionUtils.findMethod(instance.getClass(), methodName);
        return ReflectionUtils.invokeMethod(method, instance);
    }

    /**
     * 指定类的方法调用
     *
     * @param method    调用的方法
     * @param instance  类实例
     * @param paramJson 参数对象
     * @since 0.1.15-SNAPSHOT
     */
    public static void invokeMethod(Method method, @Nullable Object instance, @Nullable String paramJson) {
        // 获取参数类型
        Object[] paramTypes = getParamType(method, paramJson);
        // 执行方法
        ReflectionUtils.invokeMethod(method, instance, paramTypes);
    }

    /**
     * 指定类的方法调用
     *
     * @param className  调用的方法
     * @param methodName 类实例
     * @param paramJson  参数对象
     * @since 0.1.16-SNAPSHOT
     */
    public static void invokeMethod(String className, String methodName, String paramJson) {
        Object instance = SpringUtils.getClassBean(className);
        Assert.notNull(instance, "Required a bean of type '" + className + "' that could not be found.");
        // 无参函数
        if (StrUtil.isNullOrUndefined(paramJson)) {
            ReflectUtils.invokeMethod(instance, methodName);
        }
        // 有参函数，且仅为`String.class`
        else {
            Method method = ReflectionUtils.findMethod(instance.getClass(), methodName, String.class);
            ReflectionUtils.invokeMethod(method, instance, paramJson);
        }
    }

    /**
     * 获取指定类的所有子类/实现类
     *
     * @since 0.1.11-SNAPSHOT
     */
    public static <T> List<T> getSubTypesOf(Class<T> superClass) {
        Reflections reflections = SpringUtils.getBean(Reflections.class);
        if (ObjectUtil.isEmpty(reflections)) {
            logger.debug("Reflections Not Declaring.");
            return Collections.emptyList();
        }
        return reflections.getSubTypesOf(superClass).stream().map(SpringUtils::getBean).collect(Collectors.toList());
    }

    /**
     * 获取方法参数
     *
     * @param method    调用的方法
     * @param paramJson 参数对象
     * @return 参数列表
     * @since 0.1.15-SNAPSHOT
     */
    public static Object[] getParamType(Method method, String paramJson) {
        Type[] types = method.getGenericParameterTypes();
        if (types.length == 0) {
            return null;
        }
        return BeanUtils.toObjects(paramJson, types);
    }
}
