package com.elitesland.yst.common.base;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.yst.common.common.CloudtContextHolder;
import org.springframework.util.StringUtils;

import java.util.Map;

/**
 * <p>
 * REST API 响应码
 * </p>
 *
 * @author Michael Li
 * @date 2018-11-08
 */
public enum ApiCode {
    /**
     * Http response entity status code, corresponding to HTTP Status Code
     */
    SUCCESS(200, "操作成功"),

    UNAUTHORIZED(401, "暂未登录或token已经过期"),

    NOT_PERMISSION(403, "没有权限"),

    NOT_FOUND(404, "你请求的资源不存在"),

    FAIL(500, "操作失败"),

    VALIDATE_FAILED(404, "参数检验失败"),

    FORBIDDEN(403, "没有相关权限"),

    AUTHENTICATION_CLIENT_FORBIDDEN(401, "客户端验证失败"),

    METHOD_NOT_SUPPORT(401, "请求方式不支持"),

    LOGIN_EXCEPTION(4000, "登录失败"),

    SYSTEM_EXCEPTION(9900, "系统异常"),

    PARAMETER_EXCEPTION(9901, "请求参数校验异常"),

    PARAMETER_PARSE_EXCEPTION(9902, "请求参数解析异常"),

    HTTP_MEDIA_TYPE_EXCEPTION(9903, "HTTP Media 类型异常"),

    BUSINESS_EXCEPTION(9904, "业务处理异常"),

    DAO_EXCEPTION(9905, "数据库处理异常"),

    VERIFICATION_CODE_EXCEPTION(9906, "验证码校验异常"),

    AUTHENTICATION_EXCEPTION(9907, "登录授权异常"),

    UNAUTHENTICATED_EXCEPTION(9908, "身份认证不正确"),

    UNAUTHORIZED_EXCEPTION(9909, "没有访问权限"),

    UNAUTHORIZED_ANONYMOUS(9910, "匿名用户，未授权访问"),

    NO_USER_FOUND_EXCEPTION(9911, "用户信息未找到"),

    USER_DISABLED_EXCEPTION(9912, "用户账号已被禁用"),

    TOKEN_ACCESS_FORBIDDEN(9913, "token已被禁止访问"),

    CLOUD_SERVICE_EXCEPTION(9914, "服务调用失败"),

    CLOUD_SERVICE_UNAVAILABLE(9915, "服务不可用"),

    THREAD_OVERFLOW(9916, "线程池溢出"),

    SERVER_RESPONSE_TIMEOUT(9917, "服务器响应超时"),

    ROCKETMQ_MESSAGE_EMPTY(9918, "MQ消息主题消息为空")
    ;

    private final int code;
    private final String msg;

    ApiCode(final int code, final String msg) {
        this.code = code;
        this.msg = msg;
    }

    public static ApiCode getApiCode(int code) {
        ApiCode[] ecs = ApiCode.values();
        for (ApiCode ec : ecs) {
            if (ec.getCode() == code) {
                return ec;
            }
        }
        return SUCCESS;
    }

    public int getCode() {
        return code;
    }

    public String getMsg() {
        return msg;
    }

    public String getMsgFriendly(String errorNo) {
        switch (this) {
            case SUCCESS:
                return SUCCESS.getMsg();
            case UNAUTHORIZED:
                return "登录过期，请重新登录";
            case NOT_PERMISSION:
            case FORBIDDEN:
            case AUTHENTICATION_CLIENT_FORBIDDEN:
            case UNAUTHENTICATED_EXCEPTION:
            case UNAUTHORIZED_EXCEPTION:
            case UNAUTHORIZED_ANONYMOUS:
            case NO_USER_FOUND_EXCEPTION:
            case USER_DISABLED_EXCEPTION:
            case TOKEN_ACCESS_FORBIDDEN:
                return "没有权限";
            case NOT_FOUND:
                return "请求的内容不存在";
            case LOGIN_EXCEPTION:
            case AUTHENTICATION_EXCEPTION:
                return "登录失败";
            case VERIFICATION_CODE_EXCEPTION:
                return "验证码错误";
            case PARAMETER_PARSE_EXCEPTION:
                return "数据格式不正确";
            default:
                String ms = null;
                if (StringUtils.hasText(CloudtContextHolder.getGlobalDefaultErrorMsg())) {
                    ms = StrUtil.format(CloudtContextHolder.getGlobalDefaultErrorMsg(), Map.of("errorNo", ObjectUtil.defaultIfNull(errorNo, "001")));
                }
                return ObjectUtil.defaultIfNull(ms, "系统繁忙, 请稍后再试");
        }
    }

}
