package com.elitesland.yst.common.config;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.yst.common.common.CloudtContextHolder;
import com.elitesland.yst.common.exception.filter.ExceptionFilter;
import com.elitesland.yst.common.exception.handler.CloudtExceptionProperties;
import com.elitesland.yst.common.property.*;
import com.elitesland.yst.common.util.SpringContextHolder;
import lombok.val;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.util.Map;

/**
 * @author mir
 */
@Configuration
@EnableConfigurationProperties({IdProperties.class, CloudtExceptionProperties.class})
public class EliteslandConfig implements InitializingBean {

    @Value("${spring.application.name:#{null}}")
    private String applicationName;

    private final CloudtExceptionProperties cloudtExceptionProperties;

    public EliteslandConfig(CloudtExceptionProperties cloudtExceptionProperties) {
        this.cloudtExceptionProperties = cloudtExceptionProperties;
    }

    @Bean
    @ConfigurationProperties(prefix = "elitesland.oauth2")
    public OAuth2Properties oauth2Properties() {
        return new OAuth2Properties();
    }

    @Bean
    @ConfigurationProperties(prefix = "elitesland.service")
    public EliteslandServiceProperties eliteslandServiceProperties() {
        return new EliteslandServiceProperties();
    }

    @Bean
    @ConfigurationProperties(prefix = "elitesland.login")
    public LoginProperties loginProperties() {
        return new LoginProperties();
    }

    @Bean
    @ConfigurationProperties(prefix = "elitesland.special.account")
    public SpecialAccountProperties specialAccountProperties() {
        return new SpecialAccountProperties();
    }

    @Bean
    public SpringContextHolder springContextHolder() {
        return new SpringContextHolder();
    }

    @Bean
    public ExceptionFilter exceptionFilter() {
        return new ExceptionFilter();
    }

    @Bean
    public FilterRegistrationBean registrationBean() {
        val bean = new FilterRegistrationBean<>();
        bean.setFilter(exceptionFilter());
        bean.addUrlPatterns("/*");
        bean.setOrder(1);

        return bean;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        initCloudtContextHolder();
    }

    private void initCloudtContextHolder() {
        // 设置应用编码
        if (StringUtils.hasText(applicationName)) {
            CloudtContextHolder.setAppCode(applicationName);
        }

        // 应用名称
        String appName = getAppName();
        if (StringUtils.hasText(appName)) {
            CloudtContextHolder.setAppName(appName);
        }

        // 默认错误提示
        String defaultExceptionMsg = getDefaultMsg();
        if (StringUtils.hasText(defaultExceptionMsg)) {
            CloudtContextHolder.setGlobalDefaultErrorMsg(defaultExceptionMsg);
        }
    }

    private String getAppName() {
        String appCode = CloudtContextHolder.getAppCode();
        if (StringUtils.hasText(appCode)) {
            for (CloudtExceptionProperties.AppInfo appInfo : cloudtExceptionProperties.getGlobal().getAppInfos()) {
                if (CharSequenceUtil.equals(appCode, appInfo.getAppCode())) {
                    return appInfo.getAppName();
                }
            }
        }

        return null;
    }

    private String getDefaultMsg() {
        String appName = CloudtContextHolder.getAppName();

        String format = cloudtExceptionProperties.getGlobal().getDefaultMsgFormat();

        return StrUtil.format(format, Map.of("appName", appName));
    }
}
