package com.elitesland.yst.security.entity;

import com.elitesland.yst.common.constant.TenantType;
import com.elitesland.yst.security.common.InnerUserEnum;
import com.elitesland.yst.security.dto.SecurityOrgUserEmpBuDTO;
import com.elitesland.yst.system.dto.SysTenantDTO;
import com.elitesland.yst.system.vo.SysRoleVO;
import com.elitesland.yst.system.vo.SysUserDTO;
import lombok.Data;
import lombok.experimental.Accessors;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author Michael Li
 * @date AC 2021-04-11:16:12
 */
@Data
@Accessors(chain = true)
public class GeneralUserDetails implements UserDetails {

    private static final long serialVersionUID = 414652772107486653L;

    private SysUserDTO user;
    private String ip;
    private String browser;
    private String address;
    private LocalDateTime loginExpiredAt;
    private String token;

    /**
     * 增加返回当前用户所属的BU IDs，包括用户当前所属BU，以及这个BU
     */
    private List<Long> selfBuIds;

    /**
     * 用户补充扩展信息，主要是支撑域获取的 员工， 员工组织
     **/
    private SecurityOrgUserEmpBuDTO securityOrgUserEmpBuDTO;

    /**
     * 除用户所属的BU IDs，还有离散授权的BU IDs
     */
    private List<Long> grantedBuIds;

    /**
     * 支持业务自定义的扩展信息
     * {@link com.elitesland.yst.security.CurrentUserExtension}
     */
    private Object extendInfo;

    public GeneralUserDetails(SysUserDTO sysUserDTO) {
        this.user = sysUserDTO;
    }

    public List<String> getRoleCodes() {
        if (user.getRoles() == null) {
            return Collections.emptyList();
        }
        return user.getRoles().stream().map(SysRoleVO::getCode).collect(Collectors.toList());
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        if (user.getRoles() == null) {
            return Collections.emptyList();
        }
        return user.getRoles().stream().map(SysRoleVO::getCode)
                .map(SimpleGrantedAuthority::new).collect(Collectors.toList());
    }

    @Override
    public String getPassword() {
        return user.getPassword();
    }

    @Override
    public String getUsername() {
        return user.getUsername();
    }

    @Override
    public boolean isAccountNonExpired() {
        return !Objects.isNull(user.getEnabled()) && user.getEnabled();
    }

    @Override
    public boolean isAccountNonLocked() {
        return !Objects.isNull(user.getEnabled()) && user.getEnabled();
    }

    @Override
    public boolean isCredentialsNonExpired() {
        return !Objects.isNull(user.getEnabled()) && user.getEnabled();
    }

    @Override
    public boolean isEnabled() {
        return !Objects.isNull(user.getEnabled()) && user.getEnabled();
    }

    public Long getUserId() {
        return getUser().getId();
    }

    /**
     * 是否是系统管理员
     *
     * @return 是否是系统管理员
     */
    public boolean isSystemAdmin() {
        return InnerUserEnum.ADMIN.getUsername().equals(getUsername());
    }

    /**
     * 是否是租户管理员
     *
     * @return 是否是租户管理员
     */
    public boolean isTenantAdmin() {
        return user.getSysTenantVO() != null && Objects.equals(getUserId(), user.getSysTenantVO().getSysUserId());
    }

    /**
     * 是否是普通的租户下的用户（除了管理员）
     *
     * @return 是否是普通的租户下的用户
     */
    public boolean isTenantUser() {
        return user.getSysTenantVO() != null && !Objects.equals(getUserId(), user.getSysTenantVO().getSysUserId()) && user.getSysTenantVO().getType() != TenantType.OPERATION;
    }

    /**
     * 是否属于运营机构
     *
     * @return 是否属于运营机构
     */
    public boolean isOperation() {
        return user.getSysTenantVO() != null && user.getSysTenantVO().getType() == TenantType.OPERATION;
    }

    /**
     * 是否是普通的用户
     * <p>
     * 并非租户下的用户
     *
     * @return 是否是普通的用户
     */
    public boolean isCommonUser() {
        return user.getSysTenantVO() == null && !InnerUserEnum.ADMIN.getUsername().equals(getUsername());
    }

    public SysTenantDTO getTenant() {
        return user.getSysTenantVO();
    }
}
