package com.elitesland.cloudt.tenant.config;

import com.elitesland.cloudt.context.flyway.FlywayBuilder;
import com.elitesland.cloudt.context.flyway.FlywayCloudtAutoConfiguration;
import com.elitesland.cloudt.context.flyway.support.FlywayHelper;
import com.elitesland.cloudt.tenant.config.datasource.AbstractTenantDatasourceProvider;
import com.elitesland.cloudt.tenant.config.support.TenantContextTransfer;
import com.elitesland.cloudt.tenant.config.support.TenantRequestInterceptor;
import com.elitesland.cloudt.tenant.core.common.ConstantTenant;
import com.elitesland.cloudt.tenant.initializer.TenantSchemaInitializer;
import com.elitesland.cloudt.tenant.provider.ClientSyncTenantProvider;
import com.elitesland.cloudt.tenant.provider.TenantProvider;
import com.elitesland.cloudt.tenant.rpc.TenantRpcProvider;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.event.ApplicationContextInitializedEvent;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.core.env.Environment;
import org.springframework.core.task.TaskExecutor;
import org.springframework.lang.NonNull;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

/**
 * 租户客户端自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/24
 */
@Configuration
@ConditionalOnProperty(prefix = TenantClientProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(TenantClientProperties.class)
@AutoConfigureAfter(value = FlywayCloudtAutoConfiguration.class)
@Import({CloudtTenantClientAutoConfiguration.TenantWebServletConfiguration.class})
public class CloudtTenantClientAutoConfiguration {

    private final TenantClientProperties clientProperties;

    @DubboReference
    private TenantRpcProvider tenantRpcProvider;

    public CloudtTenantClientAutoConfiguration(TenantClientProperties clientProperties) {
        this.clientProperties = clientProperties;
    }

    @Bean
    public TenantProvider tenantProvider() {
        return new TenantProvider(tenantRpcProvider);
    }

    @Bean
    public ClientSyncTenantProvider clientSyncTenantProvider(TenantProvider tenantProvider, FlywayHelper flywayHelper, FlywayBuilder flywayBuilder) {
        return new ClientSyncTenantProvider(clientProperties, tenantProvider, flywayHelper, flywayBuilder);
    }

    @Bean
    public TenantSchemaInitializer tenantSchemaInitializer(ClientSyncTenantProvider clientSyncTenantProvider, TaskExecutor taskExecutor) {
        return new TenantSchemaInitializer(clientSyncTenantProvider, taskExecutor);
    }

    @Bean
    public TenantRequestInterceptor tenantRequestInterceptor() {
        return new TenantRequestInterceptor(tenantProvider(), clientProperties);
    }

    @Bean
    public TenantContextTransfer tenantContextTransfer() {
        return new TenantContextTransfer();
    }

    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    static class TenantWebServletConfiguration implements WebMvcConfigurer {

        private final TenantClientProperties clientProperties;
        private final TenantRequestInterceptor tenantRequestInterceptor;

        public TenantWebServletConfiguration(TenantClientProperties clientProperties, TenantRequestInterceptor tenantRequestInterceptor) {
            this.clientProperties = clientProperties;
            this.tenantRequestInterceptor = tenantRequestInterceptor;
        }

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            registry.addInterceptor(tenantRequestInterceptor).addPathPatterns(clientProperties.getPathPattern());
        }
    }

    static class TenantEnvironmentPreparedListener implements SmartApplicationListener {


        @Override
        public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
            return eventType.isAssignableFrom(ApplicationContextInitializedEvent.class);
        }

        @Override
        public void onApplicationEvent(@NonNull ApplicationEvent event) {
            if (event instanceof ApplicationContextInitializedEvent) {
                updateTenantDefaultSchema(((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment());
                return;
            }
        }

        private void updateTenantDefaultSchema(Environment environment) {
            AbstractTenantDatasourceProvider.setDefaultSchema(environment.getProperty(ConstantTenant.CONFIG_SERVICE_PREFIX + ".default-tenant-schema-name"));
            AbstractTenantDatasourceProvider.setDbUrl(environment.getProperty(TenantClientProperties.CONFIG_PREFIX + ".default-schema-db-url"));
            AbstractTenantDatasourceProvider.setDbUser(environment.getProperty(TenantClientProperties.CONFIG_PREFIX + ".default-schema-db-user"));
            AbstractTenantDatasourceProvider.setDbPassword(environment.getProperty(TenantClientProperties.CONFIG_PREFIX + ".default-schema-db-password"));

            String dbTypeStr = environment.getProperty("spring.jpa.database");
            if (!StringUtils.hasText(dbTypeStr) && StringUtils.hasText(AbstractTenantDatasourceProvider.getDbUrl())) {
                // 从dbUrl中解析
                var dbType = AbstractTenantDatasourceProvider.getDatabaseTypeByUrl(AbstractTenantDatasourceProvider.getDbUrl());
                if (dbType != null) {
                    dbTypeStr = dbType.name();
                }
            }

            if (StringUtils.hasText(dbTypeStr)) {
                try {
                    AbstractTenantDatasourceProvider.setDatabaseType(Database.valueOf(dbTypeStr.toUpperCase()));
                } catch (IllegalArgumentException e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
