package com.elitesland.cloudt.tenant.config.datasource.hibernate;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.cloudt.tenant.config.datasource.AbstractTenantDatasourceProvider;
import com.elitesland.cloudt.tenant.config.support.TenantContextHolder;
import lombok.extern.log4j.Log4j2;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.engine.jdbc.connections.internal.DriverManagerConnectionProviderImpl;
import org.hibernate.engine.jdbc.connections.spi.AbstractMultiTenantConnectionProvider;
import org.hibernate.engine.jdbc.connections.spi.ConnectionProvider;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.util.StringUtils;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

/**
 * 租户数据源提供.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/25
 */
@Log4j2
public class HibernateTenantDatasourceProvider extends AbstractMultiTenantConnectionProvider {

    private static final long serialVersionUID = -3009600657851822507L;

    private final Map<String, ConnectionProvider> connectionProviderMap;
    private final Map<String, Database> databaseMapMap;

    public HibernateTenantDatasourceProvider() {
        this.connectionProviderMap = new HashMap<>();
        databaseMapMap = new HashMap<>();
    }

    @Override
    public Connection getConnection(String tenantIdentifier) throws SQLException {
        Connection connection = super.getConnection(tenantIdentifier);
        String schema = getTenantSchema();
        Database database = getDatabaseType(tenantIdentifier);
        log.info("use schema '{}' for tenant '{}'", schema, tenantIdentifier);

        String command = AbstractTenantDatasourceProvider.generateSwitchSchemaSql(database, schema);
        try (var statement = connection.createStatement()) {
            statement.execute(command);
        }
        return connection;
    }

    @Override
    protected ConnectionProvider getAnyConnectionProvider() {
        return getConnectionProvider(TenantContextHolder.ANONYMOUS_IDENTIFIER);
    }

    @Override
    protected ConnectionProvider selectConnectionProvider(String tenantIdentifier) {
        return getConnectionProvider(tenantIdentifier);
    }

    private String getTenantSchema() {
        return Optional.ofNullable(TenantContextHolder.getCurrentTenant())
                .map(t -> {
                    if (CharSequenceUtil.isBlank(t.getSchemaName())) {
                        return null;
                    }
                    String prefix = CharSequenceUtil.isBlank(AbstractTenantDatasourceProvider.getDefaultSchema()) ?
                            "" : AbstractTenantDatasourceProvider.getDefaultSchema() + "_";
                    return prefix + t.getSchemaName();
                })
                .filter(StringUtils::hasText)
                .orElse(AbstractTenantDatasourceProvider.getDefaultSchema());
    }

    private Database getDatabaseType(String tenantIdentifier) {
        return Optional.ofNullable(tenantIdentifier)
                .map(databaseMapMap::get)
                .orElseGet(() -> {
                    var db = AbstractTenantDatasourceProvider.getDatabaseType();
                    databaseMapMap.put(tenantIdentifier, db);
                    return db;
                });
    }

    private ConnectionProvider getConnectionProvider(String tenantIdentifier) {
        return Optional.ofNullable(tenantIdentifier)
                .map(connectionProviderMap::get)
                .orElseGet(() -> createNewConnectionProvider(tenantIdentifier));
    }

    private ConnectionProvider createNewConnectionProvider(String tenantIdentifier) {
        return Optional.ofNullable(tenantIdentifier)
                .map(this::createConnectionProvider)
                .map(connectionProvider -> {
                    connectionProviderMap.put(tenantIdentifier, connectionProvider);
                    return connectionProvider;
                })
                .orElseThrow(() -> new RuntimeException("Cannot create new connection provider for tenant: " + tenantIdentifier));
    }

    private ConnectionProvider createConnectionProvider(String tenantIdentifier) {
        return Optional.ofNullable(tenantIdentifier)
                .map(this::getHibernatePropertiesForTenantId)
                .map(this::initConnectionProvider)
                .orElse(null);
    }

    private Properties getHibernatePropertiesForTenantId(String tenantId) {
        Properties properties = new Properties();
        properties.setProperty(AvailableSettings.URL, AbstractTenantDatasourceProvider.getDbUrl());
        properties.setProperty(AvailableSettings.USER, AbstractTenantDatasourceProvider.getDbUser());
        properties.setProperty(AvailableSettings.PASS, AbstractTenantDatasourceProvider.getDbPassword());

        return properties;
    }

    private ConnectionProvider initConnectionProvider(Properties hibernateProperties) {
        DriverManagerConnectionProviderImpl connectionProvider = new DriverManagerConnectionProviderImpl();
        connectionProvider.configure(hibernateProperties);
        return connectionProvider;
    }
}
