package com.elitesland.cloudt.tenant.config.support;

import com.elitesland.cloudt.tenant.core.common.ConstantTenant;
import com.elitesland.cloudt.tenant.config.TenantClientProperties;
import com.elitesland.cloudt.tenant.provider.TenantProvider;
import com.elitesland.yst.core.security.util.SecurityUtil;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.AsyncHandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 租户请求拦截器.
 * <p>
 * 主要用于设置租户上下文信息
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/26
 */
@Log4j2
public class TenantRequestInterceptor implements AsyncHandlerInterceptor {

    private final TenantProvider tenantProvider;
    private final TenantClientProperties clientProperties;

    public TenantRequestInterceptor(TenantProvider tenantProvider, TenantClientProperties clientProperties) {
        this.tenantProvider = tenantProvider;
        this.clientProperties = clientProperties;
    }

    @Override
    public boolean preHandle(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler) throws Exception {
        var tenant = analyzeSysTenant(request);
        if (tenant != null) {
            TenantContextHolder.setCurrentTenant(tenant);
        }
        return true;
    }

    @Override
    public void afterCompletion(@NonNull HttpServletRequest request, @NonNull HttpServletResponse response, @NonNull Object handler, Exception ex) throws Exception {
        TenantContextHolder.clearCurrent();
    }

    private SysTenantDTO analyzeSysTenant(HttpServletRequest request) {
        // 优先从当前用户信息中获取
        var user = SecurityUtil.getUser();
        if (user != null && user.getTenant() != null) {
            if (!StringUtils.hasText(user.getTenant().getSchemaName())) {
                return user.getTenant();
            }
            log.warn("当前租户【{}, {}】未设置schema！", user.getTenant().getId(), user.getTenant().getTenantId());
        }

        // 从请求头获取
        String tenantId = request.getHeader(ConstantTenant.HEADER_TENANT_ID);
        if (tenantId != null) {
            var tenant = tenantProvider.getById(Long.parseLong(tenantId));
            if (tenant.isPresent()) {
                return tenant.get();
            }
        }

        // 根据域名转换
        if (StringUtils.hasText(clientProperties.getTenantDomain())) {
            String domain = obtainDomain(request.getRequestURL().toString());
            if (StringUtils.hasText(domain)) {
                var tenant = tenantProvider.getByDomain(domain);
                if (tenant.isPresent()) {
                    return tenant.get();
                }
            }
        }

        return null;
    }

    private String obtainDomain(String url) {
        boolean isHttpProtocol = false;
        if (url.startsWith(ConstantTenant.PROTOCOL_HTTP)) {
            url = url.substring(7);
            isHttpProtocol = true;
        } else if (url.startsWith(ConstantTenant.PROTOCOL_HTTPS)) {
            url = url.substring(8);
            isHttpProtocol = true;
        }

        if (!isHttpProtocol) {
            return null;
        }

        int lastIndex = url.indexOf("/");
        if (lastIndex > 0) {
            url = url.substring(0, lastIndex);
        }

        // 判断是否属于我们的子域名
        url = filterChildDomain(url);

        return url;
    }

    private String filterChildDomain(String domain) {
        if (domain == null) {
            return null;
        }
        if (domain.endsWith(clientProperties.getTenantDomain())) {
            return domain.substring(0, domain.length() - clientProperties.getTenantDomain().length() - 1);
        }
        return domain;
    }
}
