package com.elitesland.cloudt.tenant.provider;

import com.elitesland.cloudt.tenant.core.common.ConstantTenant;
import com.elitesland.cloudt.tenant.rpc.TenantRpcProvider;
import com.elitesland.yst.common.common.CloudtOptional;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * 租户信息提供者.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/26
 */
@Log4j2
public class TenantProvider {

    private final TenantRpcProvider tenantRpcProvider;

    /**
     * 所有租户信息
     */
    private static final Map<String, SysTenantDTO> ALL = new HashMap<>();
    /**
     * 租户域名与租户的映射
     */
    private static final Map<String, SysTenantDTO> DOMAIN_TENANT = new HashMap<>();
    /**
     * 租户自定义域名与租户的映射
     */
    private static final Map<String, SysTenantDTO> DOMAIN_CUSTOM_TENANT = new HashMap<>();

    public TenantProvider(TenantRpcProvider tenantRpcProvider) {
        this.tenantRpcProvider = tenantRpcProvider;
    }

    /**
     * 获取所有租户信息
     *
     * @return 租户信息
     */
    @Cacheable(value = ConstantTenant.CACHE_ALL_KEY, key = "'ALL'")
    public Map<String, SysTenantDTO> getAllTenant() {
        queryTenant();

        return Collections.unmodifiableMap(ALL);
    }

    /**
     * 根据租户ID获取租户信息
     *
     * @param id 租户ID
     * @return 租户信息
     */
    @Cacheable(value = ConstantTenant.CACHE_ALL_KEY, key = "#p0")
    public CloudtOptional<SysTenantDTO> getById(Long id) {
        queryTenant();

        return CloudtOptional.of(ALL.get(id.toString()));
    }

    /**
     * 根据域名获取租户信息
     *
     * @param domain 域名
     * @return 租户信息
     */
    @Cacheable(value = ConstantTenant.CACHE_ALL_KEY, key = "#p0")
    public CloudtOptional<SysTenantDTO> getByDomain(String domain) {
        queryTenant();

        SysTenantDTO tenantDTO = DOMAIN_CUSTOM_TENANT.get(domain);
        if (tenantDTO != null) {
            return CloudtOptional.of(tenantDTO);
        }

        return CloudtOptional.of(tenantDTO);
    }

    private void queryTenant() {
        ALL.clear();
        DOMAIN_TENANT.clear();
        DOMAIN_CUSTOM_TENANT.clear();

        // 查询租户信息
        var queryTenantResult = tenantRpcProvider.allTenants();
        if (!queryTenantResult.isSuccess() || queryTenantResult.getData() == null) {
            throw new BusinessException("查询租户信息失败：" + queryTenantResult.getMsg() + "；" + queryTenantResult.getErrorMsg());
        }

        var tenants = queryTenantResult.getData();
        if (tenants.isEmpty()) {
            log.warn("暂无租户信息");
            return;
        }
        for (SysTenantDTO tenant : tenants) {
            cacheTenant(tenant);
        }
    }

    private void cacheTenant(SysTenantDTO tenant) {
        if (tenant == null) {
            return;
        }

        ALL.put(tenant.getId().toString(), tenant);

        if (StringUtils.hasText(tenant.getTenantDomain())) {
            DOMAIN_TENANT.put(tenant.getTenantName(), tenant);
        }
        if (StringUtils.hasText(tenant.getTenantCustomDomain())) {
            DOMAIN_CUSTOM_TENANT.put(tenant.getTenantCustomDomain(), tenant);
        }
    }
}
