package com.elitesland.cloudt.authorization.api.provider.config;

import com.elitesland.cloudt.authorization.api.client.AuthenticationClaim;
import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.provider.AuthenticationService;
import com.elitesland.cloudt.authorization.api.provider.config.servlet.ServletOAuth2ClientConfig;
import com.elitesland.cloudt.authorization.api.provider.config.servlet.ServletOAuth2ServerConfig;
import com.elitesland.cloudt.authorization.api.provider.config.servlet.ServletSingleConfig;
import com.elitesland.cloudt.authorization.api.provider.config.system.ConfigProperties;
import com.elitesland.cloudt.authorization.api.provider.security.AuthenticationCheckService;
import com.elitesland.cloudt.authorization.api.provider.security.impl.DefaultAuthenticationCheckServiceImpl;
import com.elitesland.cloudt.authorization.api.provider.web.controller.LoginSupportController;
import com.elitesland.yst.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.oauth2.jwt.JwtClaimsSet;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;
import org.springframework.util.Assert;

/**
 * Auth2相关自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/31
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableWebSecurity(debug = true)
@Import({ServletSingleConfig.class, ServletOAuth2ServerConfig.class, ServletOAuth2ClientConfig.class, SystemConfig.class})
@Log4j2
public class AuthorizationAutoConfiguration {

    private final AuthorizationProperties authorizationProperties;

    public AuthorizationAutoConfiguration(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
        Assert.notNull(authorizationProperties.getType(), "未知服务认证方式");

        log.info("服务认证方式：{}", authorizationProperties.getType());
    }

    /**
     * 自定义token的claim
     *
     * @return 自定义token的claim
     */
    @Bean
    public OAuth2TokenCustomizer<JwtEncodingContext> jwtCustomizer() {
        return context -> {
            Object principal = context.getPrincipal().getPrincipal();
            if (principal instanceof GeneralUserDetails) {
                GeneralUserDetails userDetails = (GeneralUserDetails) principal;
                JwtClaimsSet.Builder claims = context.getClaims();

                claims.claim(AuthenticationClaim.USERNAME, userDetails.getUsername());
                claims.claim(AuthenticationClaim.USERID, userDetails.getUser().getId());
                if (userDetails.getTenant() != null) {
                    claims.claim(AuthenticationClaim.TENANT_ID, userDetails.getTenant().getId());
                }
            }
        };
    }

    /**
     * 登录服务相关服务
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    public AuthenticationService authenticationService() {
        log.warn("未实现认证服务接口");
        return () -> null;
    }

    /**
     * 登录服务相关接口
     *
     * @param authenticationService
     * @return
     */
    @Bean
    @ConditionalOnBean({AuthenticationService.class})
    public LoginSupportController loginSupportController(AuthenticationService authenticationService) {
        return new LoginSupportController(authenticationService);
    }

    @Bean
    public AuthenticationCheckService defaultAuthenticationCheckService(ConfigProperties configProperties) {
        return new DefaultAuthenticationCheckServiceImpl(configProperties);
    }
}
