package com.elitesland.cloudt.authorization.api.provider.config;

import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitesland.cloudt.authorization.api.provider.config.system.ConfigProperties;
import com.elitesland.cloudt.authorization.api.provider.config.system.WechatProperties;
import com.elitesland.cloudt.authorization.api.provider.provider.CaptchaProvider;
import com.elitesland.cloudt.authorization.api.provider.provider.LoginSupportProvider;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.messenger.RmiMsgCarrierDubboService;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.support.RmiOrgUserEmpBuRpcService;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.system.RmiSysUserRpcService;
import com.elitesland.cloudt.authorization.api.provider.provider.sms.SmsCodeProvider;
import com.elitesland.cloudt.authorization.api.provider.provider.user.UserDetailManager;
import com.elitesland.cloudt.authorization.api.provider.provider.wechat.WechatAppletService;
import com.elitesland.cloudt.authorization.api.provider.provider.wechat.provider.WechatAppletServiceImpl;
import com.elitesland.cloudt.authorization.api.provider.web.controller.CommonLoginSupportController;
import org.apache.dubbo.config.bootstrap.DubboBootstrap;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;

/**
 * 可选登录扩展支持配置.
 * <p>
 * 根据登录认证方式选择
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/6
 */
@Import({LoginSupportConfig.ConfigOnDubbo.class})
public class LoginSupportConfig {

    private final AuthorizationProperties authorizationProperties;
    private final ConfigProperties configProperties;

    public LoginSupportConfig(AuthorizationProperties authorizationProperties, ConfigProperties configProperties) {
        this.authorizationProperties = authorizationProperties;
        this.configProperties = configProperties;
    }

    /**
     * 登录基础支持服务
     *
     * @return
     */
    @Bean
    public LoginSupportProvider loginSupportProvider() {
        return new LoginSupportProvider(configProperties);
    }

    /**
     * 登录相关基础支持服务接口
     *
     * @param loginSupportProvider
     * @return
     */
    @Bean
    public CommonLoginSupportController commonLoginSupportController(LoginSupportProvider loginSupportProvider) {
        return new CommonLoginSupportController(loginSupportProvider);
    }

    /**
     * 用户管理相关服务
     *
     * @param rmiSysUserRpcService
     * @return
     */
    @ConditionalOnBean({RmiSysUserRpcService.class})
    @Bean
    public UserDetailManager userDetailManager(RmiSysUserRpcService rmiSysUserRpcService) {
        return new UserDetailManager(authorizationProperties, configProperties, rmiSysUserRpcService);
    }

    /**
     * 短信验证码相关服务
     *
     * @param redisHelper
     * @param rmiMsgCarrierDubboService
     * @return
     */
    @ConditionalOnBean({RedisHelper.class, RmiMsgCarrierDubboService.class})
    @Bean
    public SmsCodeProvider smsCodeProvider(RedisHelper redisHelper, RmiMsgCarrierDubboService rmiMsgCarrierDubboService) {
        return new SmsCodeProvider(redisHelper, configProperties, rmiMsgCarrierDubboService);
    }

    /**
     * 验证码相关服务
     *
     * @param redisHelper
     * @return
     */
    @ConditionalOnBean({RedisHelper.class})
    @Bean
    public CaptchaProvider captchaProvider(RedisHelper redisHelper) {
        return new CaptchaProvider(redisHelper);
    }

    /**
     * 微信小程序相关服务
     *
     * @param rmiSysUserRpcService
     * @param wechatProperties
     * @return
     */
    @ConditionalOnBean({RmiSysUserRpcService.class})
    @Bean
    public WechatAppletService wechatAppletService(RmiSysUserRpcService rmiSysUserRpcService, WechatProperties wechatProperties) {
        return new WechatAppletServiceImpl(rmiSysUserRpcService, wechatProperties);
    }

    /**
     * 存在dubbo时激活相关服务
     */
    @ConditionalOnClass(DubboBootstrap.class)
    static class ConfigOnDubbo {

        /**
         * 消息服务，dubbo接口
         *
         * @return
         */
        @Bean
        public RmiMsgCarrierDubboService rmiMsgCarrierDubboService() {
            return new RmiMsgCarrierDubboService();
        }

        /**
         * 支撑域员工服务
         *
         * @return
         */
        @Bean
        public RmiOrgUserEmpBuRpcService rmiOrgUserEmpBuRpcService() {
            return new RmiOrgUserEmpBuRpcService();
        }

        /**
         * 系统域，用户服务
         *
         * @return
         */
        @Bean
        public RmiSysUserRpcService rmiSysUserRpcService() {
            return new RmiSysUserRpcService();
        }
    }
}
