package com.elitesland.cloudt.authorization.api.provider.config.servlet;

import com.elitesland.cloudt.authorization.api.client.common.SecurityConstants;
import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.client.config.security.AbstractServletSecurityConfig;
import com.elitesland.cloudt.authorization.api.client.config.security.handler.DefaultAccessDeniedHandler;
import com.elitesland.cloudt.authorization.api.client.config.security.handler.DefaultAuthenticationEntryPointHandler;
import com.elitesland.cloudt.authorization.api.client.config.security.handler.DefaultAuthenticationFailureHandler;
import com.elitesland.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitesland.cloudt.authorization.api.client.util.JwtUtil;
import com.elitesland.cloudt.authorization.api.provider.config.LoginSupportConfig;
import com.elitesland.cloudt.authorization.api.provider.security.configurer.OAuth2AuthorizationCodeStateFilterSecurityConfigurer;
import com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server.OAuth2ServerAuthenticationEntryPointHandler;
import com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server.OAuth2ServerAuthenticationSuccessHandler;
import com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server.OAuth2ServerErrorResponseHandler;
import com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server.support.OAuth2AuthorizationCodeRequestCache;
import com.elitesland.cloudt.authorization.api.provider.security.impl.RedisOAuth2AuthorizationCodeRequestCache;
import com.elitesland.cloudt.authorization.api.provider.service.impl.JpaOAuth2AuthorizationConsentService;
import com.elitesland.cloudt.authorization.api.provider.service.impl.JpaOAuth2AuthorizationService;
import com.elitesland.cloudt.authorization.api.provider.service.impl.JpaRegisteredClientRepository;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.*;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.jwk.source.JWKSource;
import com.nimbusds.jose.proc.SecurityContext;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.*;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configurers.oauth2.server.authorization.OAuth2AuthorizationServerConfigurer;
import org.springframework.security.oauth2.server.authorization.OAuth2AuthorizationConsentService;
import org.springframework.security.oauth2.server.authorization.OAuth2AuthorizationService;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClientRepository;
import org.springframework.security.oauth2.server.authorization.config.ProviderSettings;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.LoginUrlAuthenticationEntryPoint;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestHeaderRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

/**
 * OAuth2认证方式服务端.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/21
 */
@Log4j2
@ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "type", havingValue = "oauth2_server")
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@Import({LoginSupportConfig.class})
public class ServletOAuth2ServerConfig extends AbstractServletSecurityConfig {
    @Value("${spring.security.oauth2.resourceserver.jwt.issuer-uri:#{null}}")
    private String issuerUri;

    private final OAuth2AuthorizationCodeRequestCache oAuth2AuthorizationCodeRequestCache;

    public ServletOAuth2ServerConfig(AuthorizationProperties authorizationProperties, OAuth2AuthorizationCodeRequestCache oAuth2AuthorizationCodeRequestCache) {
        super(authorizationProperties);
        this.oAuth2AuthorizationCodeRequestCache = oAuth2AuthorizationCodeRequestCache;
    }

    /**
     * OAuth2 认证SecurityFilterChain
     *
     * @param http
     * @param providerSettings
     * @param clientRepository
     * @param oAuth2AuthorizationService
     * @return
     * @throws Exception
     */
    @Bean(SECURITY_CHAIN_AUTH2_SERVER)
    @Order(Ordered.HIGHEST_PRECEDENCE)
    @ConditionalOnMissingBean(name = SECURITY_CHAIN_AUTH2_SERVER)
    public SecurityFilterChain authorizationServerSecurityFilterChain(HttpSecurity http,
                                                                      ProviderSettings providerSettings,
                                                                      RegisteredClientRepository clientRepository,
                                                                      OAuth2AuthorizationService oAuth2AuthorizationService
    ) throws Exception {
        // 认证成功handler
        var authenticationSuccessHandler = new OAuth2ServerAuthenticationSuccessHandler(oAuth2AuthorizationCodeRequestCache, clientRepository, oAuth2AuthorizationService);
        // 默认失败handler
        var authenticationFailureHandler = new DefaultAuthenticationFailureHandler();
        // 未认证handler
        var entryPointHandler = new OAuth2ServerAuthenticationEntryPointHandler(oAuth2AuthorizationCodeRequestCache, providerSettings.getAuthorizationEndpoint());
        // 无权限handler
        var accessDeniedHandler = new DefaultAccessDeniedHandler();

        OAuth2AuthorizationServerConfigurer<HttpSecurity> authorizationServerConfigurer =
                new OAuth2AuthorizationServerConfigurer<>();
        RequestMatcher endpointsMatcher = authorizationServerConfigurer.getEndpointsMatcher();

        // 客户端认证配置
        authorizationServerConfigurer
                .authorizationEndpoint(configurer ->
                        configurer.errorResponseHandler(new OAuth2ServerErrorResponseHandler())
                                .authorizationResponseHandler(authenticationSuccessHandler)
                )
                .clientAuthentication(configurer ->
                        configurer.errorResponseHandler(authenticationFailureHandler)
                )
                .tokenEndpoint(configurer ->
                        configurer.errorResponseHandler(authenticationFailureHandler)
                                .accessTokenResponseHandler(authenticationSuccessHandler)
                )
        ;
        http.requestMatcher(endpointsMatcher)
                .authorizeRequests(req -> req.anyRequest().authenticated())
                .apply(authorizationServerConfigurer)
                .and()
                // 对于不支持redirect的终端，依赖于state维持状态的配置
                .apply(new OAuth2AuthorizationCodeStateFilterSecurityConfigurer<>())
                .and()
                // 异常处理配置
                .exceptionHandling(configurer -> {
                    // 未认证时的处理
                    if (StringUtils.hasText(authorizationProperties.getLoginPage())) {
                        // 有配置登录页，则针对web型的进行支持重定向至登录页
                        configurer.defaultAuthenticationEntryPointFor(entryPointHandler, new RequestHeaderRequestMatcher(SecurityConstants.HEADER_AUTH_REDIRECT, "false"))
                                .defaultAuthenticationEntryPointFor(new LoginUrlAuthenticationEntryPoint(authorizationProperties.getLoginPage()), new AntPathRequestMatcher("/**"));
                    } else {
                        // 未配置登录页，则都直接json格式返回
                        configurer.authenticationEntryPoint(entryPointHandler);
                    }

                    // 无权限时的处理
                    configurer.accessDeniedHandler(accessDeniedHandler);
                })
        ;

        // csrf配置
        if (Boolean.FALSE.equals(authorizationProperties.getCsrfEnabled())) {
            // 关闭csrf
            http.csrf().disable();
        } else {
            http.csrf().ignoringRequestMatchers(endpointsMatcher);
        }
        // cors配置
        corsConfiguration(http);

        return http.build();
    }

    /**
     * 用户认证SecurityFilterChain
     *
     * @param http
     * @return
     * @throws Exception
     */
    @Bean(SECURITY_CHAIN_DEFAULT)
    @ConditionalOnMissingBean(name = SECURITY_CHAIN_DEFAULT)
    public SecurityFilterChain defaultSecurityFilterChain(HttpSecurity http) throws Exception {
        defaultSecurityConfig(http);

        // resource server配置
        http.oauth2ResourceServer(config ->
                config.jwt()
                        .and()
                        .authenticationEntryPoint(new DefaultAuthenticationEntryPointHandler(authorizationProperties.getLoginPage()))
        );

        return http.build();
    }

    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnBean(RedisHelper.class)
    public OAuth2AuthorizationCodeRequestCache oAuth2AuthorizationCodeRequestCache(RedisHelper redisHelper) {
        return new RedisOAuth2AuthorizationCodeRequestCache(redisHelper);
    }

    /**
     * jwk源配置
     *
     * @param rsaKey
     * @return
     */
    @Bean
    public JWKSource<SecurityContext> jwkSource(RSAKey rsaKey) {
        return JwtUtil.generateJwkSource(rsaKey);
    }

    /**
     * 配置
     *
     * @return
     */
    @Bean
    public ProviderSettings providerSettings() {
        Assert.hasText(issuerUri, "spring.security.oauth2.resourceserver.jwt.issuer-uri不可为空");
        return ProviderSettings.builder().issuer(issuerUri).build();
    }

    /**
     * 有jpa时的配置
     */
    @ConditionalOnClass(JpaRepository.class)
    static class ConfigOnJpa {
        /**
         * 客户端 持久化接口
         *
         * @param oAuth2RegisteredClientRepo
         * @param oAuth2RegisteredClientRepoProc
         * @return
         */
        @Bean
        @ConditionalOnMissingBean
        public RegisteredClientRepository registeredClientRepository(OAuth2RegisteredClientRepo oAuth2RegisteredClientRepo,
                                                                     OAuth2RegisteredClientRepoProc oAuth2RegisteredClientRepoProc) {
            return new JpaRegisteredClientRepository(oAuth2RegisteredClientRepo, oAuth2RegisteredClientRepoProc);
        }

        /**
         * authorization令牌 服务接口
         *
         * @param registeredClientRepository
         * @param oAuth2AuthenticationRepo
         * @param oAuth2AuthenticationRepoProc
         * @return
         */
        @Bean
        @ConditionalOnMissingBean
        public OAuth2AuthorizationService authorizationService(RegisteredClientRepository registeredClientRepository,
                                                               OAuth2AuthenticationRepo oAuth2AuthenticationRepo,
                                                               OAuth2AuthenticationRepoProc oAuth2AuthenticationRepoProc) {
            return new JpaOAuth2AuthorizationService(registeredClientRepository, oAuth2AuthenticationRepo, oAuth2AuthenticationRepoProc);
        }

        /**
         * 自定义授权内容服务
         *
         * @param registeredClientRepository
         * @param oAuth2AuthorizationConsentRepo
         * @param oAuth2AuthorizationConsentRepoProc
         * @return
         */
        @Bean
        @ConditionalOnMissingBean
        public OAuth2AuthorizationConsentService authorizationConsentService(RegisteredClientRepository registeredClientRepository,
                                                                             OAuth2AuthorizationConsentRepo oAuth2AuthorizationConsentRepo,
                                                                             OAuth2AuthorizationConsentRepoProc oAuth2AuthorizationConsentRepoProc) {
            return new JpaOAuth2AuthorizationConsentService(registeredClientRepository, oAuth2AuthorizationConsentRepo, oAuth2AuthorizationConsentRepoProc);
        }
    }
}
