package com.elitesland.cloudt.authorization.api.provider.config.servlet;

import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.client.config.security.AbstractServletSecurityConfig;
import com.elitesland.cloudt.authorization.api.client.config.security.handler.DefaultAuthenticationFailureHandler;
import com.elitesland.cloudt.authorization.api.client.util.JwtUtil;
import com.elitesland.cloudt.authorization.api.provider.config.LoginSupportConfig;
import com.elitesland.cloudt.authorization.api.provider.security.configurer.LoginFilterSecurityConfigurer;
import com.elitesland.cloudt.authorization.api.provider.security.configurer.support.LoginFilterCustomizer;
import com.elitesland.cloudt.authorization.api.provider.security.handler.JwtAuthenticationSuccessHandler;
import com.nimbusds.jose.jwk.RSAKey;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;
import org.springframework.security.web.SecurityFilterChain;

/**
 * 单体认证方式.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
@Log4j2
@ConditionalOnProperty(prefix = AuthorizationProperties.CONFIG_PREFIX, name = "type", havingValue = "single")
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
@Import({LoginSupportConfig.class})
public class ServletSingleConfig extends AbstractServletSecurityConfig {
    private final RSAKey rsaKey;
    private ObjectProvider<OAuth2TokenCustomizer<JwtEncodingContext>> tokenCustomizerObjectProvider;
    private ObjectProvider<LoginFilterCustomizer<HttpSecurity>> loginFilterCustomizers;

    public ServletSingleConfig(AuthorizationProperties authorizationProperties, RSAKey rsaKey) {
        super(authorizationProperties);
        this.rsaKey = rsaKey;
    }

    @Bean(SECURITY_CHAIN_DEFAULT)
    @ConditionalOnMissingBean(name = SECURITY_CHAIN_DEFAULT)
    public SecurityFilterChain defaultSecurityFilterChain(HttpSecurity http) throws Exception {
        var successHandler = new JwtAuthenticationSuccessHandler(authorizationProperties, cacheObjectProvider.getIfAvailable(),
                JwtUtil.buildJwtEncoder(rsaKey));

        // token自定义处理
        var tokenCustomizer = tokenCustomizerObjectProvider.getIfAvailable();
        successHandler.setTokenCustomizer(tokenCustomizer);

        super.defaultSecurityConfig(http)
                .apply(new LoginFilterSecurityConfigurer<>(loginFilterCustomizers))
                .successHandler(successHandler)
                .failureHandler(new DefaultAuthenticationFailureHandler())
        ;

        return http.build();
    }

    @Autowired
    public void setTokenCustomizerObjectProvider(ObjectProvider<OAuth2TokenCustomizer<JwtEncodingContext>> tokenCustomizerObjectProvider) {
        this.tokenCustomizerObjectProvider = tokenCustomizerObjectProvider;
    }

    @Autowired
    public void setLoginFilterCustomizers(ObjectProvider<LoginFilterCustomizer<HttpSecurity>> loginFilterCustomizers) {
        this.loginFilterCustomizers = loginFilterCustomizers;
    }
}
