package com.elitesland.cloudt.authorization.api.provider.provider;

import cn.hutool.core.util.PhoneUtil;
import com.elitesland.cloudt.authorization.api.provider.config.system.*;
import com.elitesland.cloudt.authorization.api.provider.model.vo.resp.CaptchaRespVO;
import com.elitesland.cloudt.authorization.api.provider.provider.sms.SmsCodeProvider;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.common.exception.BusinessException;
import lombok.extern.log4j.Log4j2;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;

import java.time.Duration;

/**
 * 通用登录相关扩展支撑.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/08
 */
@Log4j2
public class LoginSupportProvider {

    private final ConfigProperties configProperties;

    private CaptchaProvider captchaManager;
    private SmsCodeProvider smsCodeManager;

    public LoginSupportProvider(ConfigProperties configProperties) {
        this.configProperties = configProperties;
    }

    /**
     * 生成验证码
     *
     * @return 验证码
     */
    public ApiResult<CaptchaRespVO> generateCaptcha() {
        LoginProperties loginProperties = configProperties.getLogin();
        CaptchaInfo captchaInfo = loginProperties.getCaptchaInfo();
        CaptchaRespVO respVO = null;

        // 验证码的有效期
        var timeToLive = Duration.ofMinutes(captchaInfo.getExpiration());

        val captchaHu = loginProperties.getCaptchaHu();
        if (captchaHu != null) {
            respVO = getCaptchaManager().generateCaptchaHu(CaptchaEnum.LOGIN, captchaHu, timeToLive);
        } else {
            respVO = getCaptchaManager().generateCaptcha(CaptchaEnum.LOGIN, loginProperties.getCaptcha(), timeToLive);
        }
        return ApiResult.ok(respVO);
    }

    /**
     * 校验验证码
     *
     * @param captchaKey  验证码key
     * @param captchaText 验证码内容
     * @return 校验结果
     */
    public ApiResult<Boolean> verifyCaptcha(String captchaKey, String captchaText) {
        if (!StringUtils.hasText(captchaKey)) {
            return ApiResult.fail("未设置验证码");
        }

        if (!StringUtils.hasText(captchaText)) {
            return ApiResult.fail("验证码为空");
        }

        boolean verify = getCaptchaManager().verifyCaptcha(CaptchaEnum.LOGIN, captchaKey, captchaText);
        return ApiResult.ok(verify);
    }

    /**
     * 发送短信验证码
     *
     * @param mobile 手机号
     * @return 发送结果
     */
    public ApiResult<String> sendSmsCode(String mobile) {
        if (!StringUtils.hasText(mobile)) {
            return ApiResult.fail("手机号为空");
        }
        if (!PhoneUtil.isMobile(mobile)) {
            return ApiResult.fail("手机号格式不正确");
        }

        String code = getSmsCodeManager().send(SmsCodeEnum.LOGIN, mobile, null, null);

        String msg = code == null ? "发送成功" : "模拟测试短信验证码，验证码是" + code;
        return ApiResult.ok(msg);
    }

    public CaptchaProvider getCaptchaManager() {
        if (captchaManager == null) {
            throw new BusinessException("验证码服务未初始化");
        }
        return captchaManager;
    }

    @Autowired(required = false)
    public void setCaptchaManager(CaptchaProvider captchaManager) {
        this.captchaManager = captchaManager;
    }

    public SmsCodeProvider getSmsCodeManager() {
        if (smsCodeManager == null) {
            throw new BusinessException("短信服务未初始化");
        }
        return smsCodeManager;
    }

    @Autowired(required = false)
    public void setSmsCodeManager(SmsCodeProvider smsCodeManager) {
        this.smsCodeManager = smsCodeManager;
    }
}
