package com.elitesland.cloudt.authorization.api.provider.provider.oauth2.client;

import com.elitesland.cloudt.authorization.api.client.model.OAuthToken;
import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.provider.model.bo.OAuth2ClientConfigBO;
import com.elitesland.yst.common.base.ApiResult;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.log4j.Log4j2;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.lang.NonNull;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.OAuth2AuthorizationServerMetadataClaimNames;
import org.springframework.security.oauth2.core.endpoint.OAuth2ParameterNames;
import org.springframework.security.oauth2.core.endpoint.PkceParameterNames;
import org.springframework.util.Assert;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Map;

/**
 * OAuth2客户端服务提供者.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/5
 */
@Log4j2
public class OAuth2ClientProvider implements ApplicationRunner {

    private final AuthorizationProperties authorizationProperties;
    private OAuth2ClientConfigBO oAuth2ClientConfigBO;
    private RestTemplate restTemplate;

    public OAuth2ClientProvider(AuthorizationProperties authorizationProperties) {
        this.authorizationProperties = authorizationProperties;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        initConfig();
    }

    /**
     * 获取客户端配置
     *
     * @return 客户端配置
     */
    public OAuth2ClientConfigBO getoAuth2ClientConfigBO() {
        return oAuth2ClientConfigBO.copy();
    }

    /**
     * 授权码换取token
     *
     * @param clientId     客户端ID
     * @param code         授权码
     * @param redirectUri  重定向
     * @param codeVerifier
     * @return
     */
    public ApiResult<OAuthToken> code2AccessToken(@NonNull String clientId, @NonNull String code, @NonNull String redirectUri, String codeVerifier) {
        var clientConfig = getoAuth2ClientConfigBO();

        // 组织请求参数
        MultiValueMap<String, Object> postParam = new LinkedMultiValueMap<>(8);
        postParam.add(OAuth2ParameterNames.CLIENT_ID, clientConfig.getClientId());
        postParam.add(OAuth2ParameterNames.CLIENT_SECRET, clientConfig.getClientSecret());
        postParam.add(OAuth2ParameterNames.GRANT_TYPE, AuthorizationGrantType.AUTHORIZATION_CODE.getValue());
        postParam.add(OAuth2ParameterNames.CODE, code);
        postParam.add(OAuth2ParameterNames.REDIRECT_URI, redirectUri);
        if (StringUtils.hasText(codeVerifier)) {
            postParam.add(PkceParameterNames.CODE_VERIFIER, codeVerifier);
        }

        try {
            var resp = restTemplate.exchange("/oauth2/token",
                    HttpMethod.POST, new HttpEntity<>(postParam), new ParameterizedTypeReference<ApiResult<OAuthToken>>() {
            });
            if (resp.getStatusCode().is2xxSuccessful()) {
                return resp.getBody();
            }
            log.error("授权码转token失败：{}", resp.getStatusCode());
            return ApiResult.fail("获取认证token失败");
        } catch (Exception e) {
            log.error("获取认证token失败：", e);
            return ApiResult.fail("获取认证token失败！");
        }
    }

    private void initConfig() {
        var client = authorizationProperties.getOauth2Client();
        Assert.hasText(client.getClientId(), "客户端ID为空");
        Assert.hasText(client.getClientSecret(), "客户端secret为空");
        Assert.hasText(client.getServerAddress(), "OAuth2服务端地址为空");

        // 构建restTemplate工具
        restTemplate = buildRestTemplate();

        OAuth2ClientConfigBO configBO = new OAuth2ClientConfigBO();
        configBO.setClientId(client.getClientId());
        configBO.setClientSecret(client.getClientSecret());

        Map<String, Object> serverConfig = null;
        if (StringUtils.hasText(client.getAuthorizeEndpoint())) {
            configBO.setAuthorizeEndpoint(client.getAuthorizeEndpoint());
        } else {
            // 调用auth服务查询
            if (serverConfig == null) {
                serverConfig = queryServerConfig();
            }
            String authorizeEndpoint = (String) serverConfig.get(OAuth2AuthorizationServerMetadataClaimNames.AUTHORIZATION_ENDPOINT);
            Assert.hasText(authorizeEndpoint, "OAuth2客户端的authorizeEndpoint未配置");
            configBO.setAuthorizeEndpoint(authorizeEndpoint);
        }

        if (StringUtils.hasText(client.getTokenEndpoint())) {
            configBO.setTokenEndpoint(client.getTokenEndpoint());
        } else {
            // 调用auth服务查询
            if (serverConfig == null) {
                serverConfig = queryServerConfig();
            }
            String tokenEndpoint = (String) serverConfig.get(OAuth2AuthorizationServerMetadataClaimNames.TOKEN_ENDPOINT);
            Assert.hasText(tokenEndpoint, "OAuth2客户端的tokenEndpoint未配置");
            configBO.setTokenEndpoint(tokenEndpoint);
        }

        oAuth2ClientConfigBO = configBO;
    }

    private RestTemplate buildRestTemplate() {
        return new RestTemplateBuilder()
                .requestFactory(this::getClientHttpRequestFactory)
                .rootUri(authorizationProperties.getOauth2Client().getServerAddress())
                .customizers(restTemplateCustomizer())
                .build();
    }

    private RestTemplateCustomizer restTemplateCustomizer() {
        var objectMapper = new ObjectMapper();

        var javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")));
        objectMapper.registerModule(javaTimeModule);

        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

        return template -> {
            for (var convert : template.getMessageConverters()) {
                if (convert instanceof MappingJackson2HttpMessageConverter) {
                    ((MappingJackson2HttpMessageConverter) convert).setObjectMapper(objectMapper);
                    return;
                }
            }
            template.getMessageConverters().add(new MappingJackson2HttpMessageConverter(objectMapper));
        };
    }

    private ClientHttpRequestFactory getClientHttpRequestFactory() {
        SSLConnectionSocketFactory sslConnectionSocketFactory = null;

        try {
            SSLContextBuilder sslContextBuilder = new SSLContextBuilder();
            sslContextBuilder.loadTrustMaterial(null, new TrustSelfSignedStrategy());
            sslConnectionSocketFactory = new SSLConnectionSocketFactory(sslContextBuilder.build(), NoopHostnameVerifier.INSTANCE);
        } catch (Exception e) {
            e.printStackTrace();
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom();
        httpClientBuilder.setSSLSocketFactory(sslConnectionSocketFactory);

        var requestFactory = new HttpComponentsClientHttpRequestFactory();
        requestFactory.setHttpClient(httpClientBuilder.build());
        requestFactory.setConnectTimeout((int) Duration.ofSeconds(10).toMillis());
        requestFactory.setReadTimeout((int) Duration.ofSeconds(30).toMillis());
        return requestFactory;
    }

    private Map<String, Object> queryServerConfig() {
        try {
            var resp = restTemplate.exchange("/.well-known/oauth-authorization-server", HttpMethod.GET, null, new ParameterizedTypeReference<Map<String, Object>>() {
            });
            if (resp.getStatusCode().is2xxSuccessful()) {
                log.info("查询OAuth2服务端配置成功：{}", resp.getBody());
                return resp.getBody();
            }
            log.warn("查询OAuth2服务端配置失败：{}", resp.getStatusCode());
        } catch (Exception e) {
            log.error("查询OAuth2服务端配置异常", e);
        }
        return Collections.emptyMap();
    }
}
