package com.elitesland.cloudt.authorization.api.provider.provider.sms;

import cn.hutool.core.util.RandomUtil;
import com.elitesland.cloudt.authorization.api.client.tool.RedisHelper;
import com.elitesland.cloudt.authorization.api.provider.config.system.ConfigProperties;
import com.elitesland.cloudt.authorization.api.provider.config.system.Constants;
import com.elitesland.cloudt.authorization.api.provider.config.system.SmsCodeEnum;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.messenger.RmiMsgCarrierDubboService;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.coord.messenger.sender.provider.param.GeneralDubboCarrier;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * 短信验证码服务.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/09
 */
@Log4j2
public class SmsCodeProvider {

    private final RedisHelper redisHelper;
    private final ConfigProperties configProperties;
    private final RmiMsgCarrierDubboService rmiMsgCarrierDubboService;

    public SmsCodeProvider(RedisHelper redisHelper, ConfigProperties configProperties, RmiMsgCarrierDubboService rmiMsgCarrierDubboService) {
        this.redisHelper = redisHelper;
        this.configProperties = configProperties;
        this.rmiMsgCarrierDubboService = rmiMsgCarrierDubboService;
    }

    /**
     * 发送验证码
     *
     * @param smsCode 验证码类型
     * @param mobile  手机号
     * @param code    验证码
     * @param content 短信内容，为空的话默认取值{@link ConfigProperties.Sms#formatContent}
     * @return 测试时返回验证码
     */
    public String send(SmsCodeEnum smsCode, String mobile, String code, String content) {
        Assert.notNull(smsCode, "未知验证码类型");
        Assert.hasText(mobile, "手机号不能为空");

        String key = buildCacheKey(smsCode, mobile);
        String value = null;
        try {
            value = (String) redisHelper.execute(redisUtils -> redisUtils.get(key));
        } catch (Exception e) {
            throw new BusinessException("生成验证码失败，redis服务异常");
        }
        if (value != null) {
            throw new BusinessException("验证码发送过于频繁");
        }

        var sms = configProperties.getSms();

        // 生成验证码和短信内容
        if (!StringUtils.hasText(code)) {
            code = RandomUtil.randomStringUpper(sms.getCaptchaLength());
        }
        if (!StringUtils.hasText(content)) {
            content = sms.getFormatContent();
        }
        long ttl = sms.getTtl().toSeconds();
        content = MessageFormat.format(content, code, ttl);

        // 验证码加入缓存
        String finalCode = code;
        try {
            redisHelper.execute(redisUtils -> {
                redisUtils.set(key, finalCode, ttl);
                return null;
            });
        } catch (Exception e) {
            throw new BusinessException("生成验证码失败，redis服务异常");
        }

        boolean test = sms.getTest() != null && sms.getTest();
        if (!test) {
            // 非测试环境，则发送短信
            List<GeneralDubboCarrier> carriers = new ArrayList<>();
            GeneralDubboCarrier generalDubboCarrier = new GeneralDubboCarrier();
            generalDubboCarrier.setTos(new String[]{mobile});
            generalDubboCarrier.setPureContent(content);
            generalDubboCarrier.setIsRichContent(false);
            generalDubboCarrier.setTransType("SMS");
            carriers.add(generalDubboCarrier);
            ApiResult<Object> sendResult = null;
            try {
                sendResult = rmiMsgCarrierDubboService.generalSend(carriers);
            } catch (Exception e) {
                log.error("短信发送失败：", e);
            }
            if (sendResult == null || !sendResult.isSuccess()) {
                throw new BusinessException("短信发送失败");
            }

            return null;
        }
        return code;
    }

    /**
     * 校验验证码
     *
     * @param smsCode 验证码类型
     * @param mobile  手机号
     * @param code    验证码
     * @return 是否校验通过
     */
    public boolean verify(SmsCodeEnum smsCode, String mobile, String code) {
        Assert.notNull(smsCode, "未知验证码类型");
        Assert.hasText(mobile, "手机号为空");
        Assert.hasText(code, "验证码为空");

        String key = buildCacheKey(smsCode, mobile);

        try {
            return redisHelper.execute(redisUtils -> {
                Object theCode = redisUtils.get(key);
                boolean verified = code.equals(theCode);
                if (verified) {
                    try {
                        redisUtils.del(key);
                    } catch (Exception e) {
                        log.error("删除校验通过的验证码【" + key + "】失败：", e);
                    }
                }
                return verified;
            });
        } catch (Exception e) {
            throw new BusinessException("验证码校验失败");
        }
    }

    private String buildCacheKey(SmsCodeEnum smsCode, String mobile) {
        return Constants.CACHE_KEY_PREFIX + "SMS_CODE:" + smsCode + ":" + mobile;
    }

}
