package com.elitesland.cloudt.authorization.api.provider.provider.user;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.BooleanUtil;
import com.elitesland.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.provider.config.system.ConfigProperties;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.support.RmiOrgUserEmpBuRpcService;
import com.elitesland.cloudt.authorization.api.provider.provider.rmi.system.RmiSysUserRpcService;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.security.dto.SecurityOrgUserEmpBuDTO;
import com.elitesland.yst.security.entity.GeneralUserDetails;
import com.elitesland.yst.supportdomain.provider.org.dto.OrgUserEmpBuDTO;
import com.elitesland.yst.system.vo.SysRoleVO;
import com.elitesland.yst.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import java.util.function.Supplier;

/**
 * 登录用户管理.
 * <p>
 * 用以登录时获取用户相关信息
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/06
 */
@Log4j2
public class UserDetailManager {

    private final AuthorizationProperties authorizationProperties;
    private final ConfigProperties configProperties;
    private final RmiSysUserRpcService rmiSysUserRpcService;
    private RmiOrgUserEmpBuRpcService rmiOrgUserEmpBuRpcService;

    public UserDetailManager(AuthorizationProperties authorizationProperties, ConfigProperties configProperties,
                             RmiSysUserRpcService rmiSysUserRpcService) {
        this.authorizationProperties = authorizationProperties;
        this.configProperties = configProperties;
        this.rmiSysUserRpcService = rmiSysUserRpcService;
    }

    /**
     * 根据用户名加载用户信息
     *
     * @param username 用户名
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByUsername(String username) {
        return loadUser(() -> rmiSysUserRpcService.getUserByUsername(username).getData(), "账号或密码错误");
    }

    /**
     * 根据手机号加载用户信息
     *
     * @param mobile 手机号
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByMobile(String mobile) {
        return loadUser(() -> rmiSysUserRpcService.getUserByMobile(mobile).getData(), "手机号或密码错误");
    }

    /**
     * 根据邮箱加载用户信息
     *
     * @param email 邮箱
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByEmail(String email) {
        return loadUser(() -> rmiSysUserRpcService.getUserByEmail(email).getData(), "邮箱或密码错误");
    }

    /**
     * 根据账号加载用户信息
     *
     * @param account 账号
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByAccount(String account) {
        return loadUser(() -> rmiSysUserRpcService.getUserByAccount(account).getData(), "账号或密码错误");
    }

    /**
     * 根据用户openid加载用户信息
     *
     * @param wechatOpenid 微信openId
     * @return 用户信息
     */
    public GeneralUserDetails loadUserByWechatOpenid(String wechatOpenid) {
        // 查询用户信息
        SysUserDTO user = rmiSysUserRpcService.getUserByWechatOpenid(wechatOpenid).getData();
        if (user == null) {
            throw new AuthorizationException("微信号暂未绑定用户");
        }
        checkSysUserStatus(user);

        // 查询职员信息
        SecurityOrgUserEmpBuDTO empBu = findUserEmpBuInfo(user.getId());

        return wrapUserDetail(user, empBu);
    }

    @Autowired(required = false)
    public void setRmiOrgUserEmpBuRpcService(RmiOrgUserEmpBuRpcService rmiOrgUserEmpBuRpcService) {
        this.rmiOrgUserEmpBuRpcService = rmiOrgUserEmpBuRpcService;
    }

    private GeneralUserDetails loadUser(Supplier<SysUserDTO> userDTOSupplier, String userNotExistsTip) {
        // 查询用户信息
        SysUserDTO user = userDTOSupplier.get();
        if (user == null) {
            throw new UsernameNotFoundException(userNotExistsTip);
        }
        checkSysUserStatus(user);

        // 查询职员信息
        SecurityOrgUserEmpBuDTO empBu = findUserEmpBuInfo(user.getId());

        return wrapUserDetail(user, empBu);
    }

    private void checkSysUserStatus(SysUserDTO user) {
        if (!BooleanUtil.isTrue(user.getEnabled())) {
            throw new DisabledException("账号已禁用");
        }
    }

    private GeneralUserDetails wrapUserDetail(SysUserDTO user, SecurityOrgUserEmpBuDTO empBu) {
        // 处理角色编码
        if (CollUtil.isNotEmpty(user.getRoles())) {
            for (SysRoleVO role : user.getRoles()) {
                if (role.getCode() == null || role.getCode().startsWith(authorizationProperties.getRolePrefix())) {
                    continue;
                }
                role.setCode(authorizationProperties.getRolePrefix() + role.getCode());
            }
        }

        return new GeneralUserDetails(user)
                .setSecurityOrgUserEmpBuDTO(empBu);
    }

    private SecurityOrgUserEmpBuDTO findUserEmpBuInfo(Long sysUserId) {
        // 是否必须有支撑域的员工信息
        boolean requiredSupportEmployee = configProperties.getLogin().getRequiredSupportEmployee();

        ApiResult<OrgUserEmpBuDTO> queryResult = null;
        try {
            queryResult = rmiOrgUserEmpBuRpcService.findUserEmpBuInfoByUserId(sysUserId);
        } catch (Exception e) {
            if (requiredSupportEmployee) {
                throw new RuntimeException(e);
            }

            return null;
        }

        if (queryResult == null || !queryResult.isSuccess()) {
            if (requiredSupportEmployee) {
                throw new AuthenticationServiceException("查询员工信息失败，请稍后再试");
            }
            return null;
        }

        SecurityOrgUserEmpBuDTO empBu = new SecurityOrgUserEmpBuDTO();

        OrgUserEmpBuDTO userEmpBuDTO = queryResult.getData();
        empBu.setUserId(sysUserId);

        if (userEmpBuDTO == null) {
            return empBu;
        }
        empBu.setUserId(userEmpBuDTO.getUserId());
        empBu.setEmpId(userEmpBuDTO.getEmpId());
        empBu.setEmpCodePath(userEmpBuDTO.getEmpCodePath());
        empBu.setChildEmpIds(userEmpBuDTO.getChildEmpIds());
        empBu.setEmpBuTreeId(userEmpBuDTO.getEmpBuTreeId());
        empBu.setEmpBuIds(userEmpBuDTO.getEmpBuIds());
        empBu.setEmpBuCodePath(userEmpBuDTO.getEmpBuCodePath());
        empBu.setChildEmpBuIds(userEmpBuDTO.getChildEmpBuIds());
        empBu.setTopBuCode(userEmpBuDTO.getHeadOuCode());
        empBu.setTopBuName(userEmpBuDTO.getHeadOuName());

        return empBu;
    }
}
