package com.elitesland.cloudt.authorization.api.provider.security.configurer;

import com.elitesland.cloudt.authorization.api.provider.security.configurer.support.LoginFilterCustomizer;
import com.elitesland.cloudt.authorization.api.provider.security.grant.AbstractCustomAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.CustomGrantAuthenticationFilter;
import com.elitesland.cloudt.authorization.api.provider.security.grant.account_pwd.AccountPasswordAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.client_user.ClientUserAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.mobile_pwd.MobilePasswordAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.mobile_sms.MobileSmsAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.password.UsernamePasswordAuthenticationProvider;
import com.elitesland.cloudt.authorization.api.provider.security.grant.wx_minapp.WechatOpenidAuthenticationProvider;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.lang.NonNull;
import org.springframework.security.config.annotation.web.HttpSecurityBuilder;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.List;

/**
 * 自定义登录过滤器配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/27
 */
@Log4j2
public class LoginFilterSecurityConfigurer<H extends HttpSecurityBuilder<H>>
        extends AbstractCustomAuthenticationFilterConfigurer<H, LoginFilterSecurityConfigurer<H>, CustomGrantAuthenticationFilter> {

    private String loginProcessingUrl = CustomGrantAuthenticationFilter.DEFAULT_FILTER_PROCESS_URI;
    private final List<AbstractCustomAuthenticationProvider<?>> customAuthenticationProviders = new ArrayList<>();
    private final ObjectProvider<LoginFilterCustomizer<H>> objectProvider;

    public LoginFilterSecurityConfigurer(ObjectProvider<LoginFilterCustomizer<H>> objectProvider) {
        this.objectProvider = objectProvider;
    }

    /**
     * 添加自定义AuthenticationProvider
     *
     * @param authenticationProviders authenticationProvider
     * @return
     */
    public LoginFilterSecurityConfigurer<H> addCustomAuthenticationProvider(@NonNull List<AbstractCustomAuthenticationProvider<?>> authenticationProviders) {
        customAuthenticationProviders.addAll(authenticationProviders);
        return this;
    }

    @Override
    public void init(H http) throws Exception {
        // 配置Filter
        CustomGrantAuthenticationFilter authenticationFilter = new CustomGrantAuthenticationFilter(this.loginProcessingUrl);
        this.setAuthenticationFilter(authenticationFilter);
        super.loginProcessingUrl(loginProcessingUrl);

        // 自定义扩展处理
        objectProvider.stream().forEach(loginFilterCustomizer -> loginFilterCustomizer.customizer(this));

        // AuthenticationProvider
        if (this.customAuthenticationProviders.isEmpty()) {
            customAuthenticationProviders.addAll(defaultAuthenticationProviders());
        }

        for (AbstractCustomAuthenticationProvider<?> customAuthenticationProvider : customAuthenticationProviders) {
            // 配置filter的TokenConvert
            var tokenConstructor = customAuthenticationProvider.getAuthenticationTokenType().getConstructor();
            authenticationFilter.addAuthenticationTokenConvert(tokenConstructor.newInstance());

            // 添加AuthenticationProvider
            http.authenticationProvider(this.postProcess(customAuthenticationProvider));
        }

        super.init(http);
    }

    @Override
    public void configure(H http) throws Exception {
        getAuthenticationFilter().setFilterProcessesUrl(this.loginProcessingUrl);

        super.configure(http);
    }

    @Override
    protected RequestMatcher createLoginProcessingUrlMatcher(String loginProcessingUrl) {
        return new AntPathRequestMatcher(loginProcessingUrl);
    }

    @Override
    public LoginFilterSecurityConfigurer<H> loginProcessingUrl(String loginProcessingUrl) {
        Assert.hasText(loginProcessingUrl, "loginProcessingUrl cannot be empty");
        this.loginProcessingUrl = loginProcessingUrl;
        return this;
    }

    public List<AbstractCustomAuthenticationProvider<?>> defaultAuthenticationProviders() {
        return List.of(new AccountPasswordAuthenticationProvider(), new ClientUserAuthenticationProvider(), new MobilePasswordAuthenticationProvider()
                , new MobileSmsAuthenticationProvider(), new UsernamePasswordAuthenticationProvider(), new WechatOpenidAuthenticationProvider());
    }
}
