package com.elitesland.cloudt.authorization.api.provider.security.grant;

import com.elitesland.cloudt.authorization.api.provider.common.CustomOAuth2ParameterNames;
import com.elitesland.cloudt.authorization.api.provider.common.LoginType;
import org.springframework.http.HttpMethod;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * 自定义授权登录Filter.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/27
 */
public class CustomGrantAuthenticationFilter extends AbstractAuthenticationProcessingFilter {

    /**
     * 默认登录路径
     */
    public static final String DEFAULT_FILTER_PROCESS_URI = "/oauth/login";
    private final Map<String, AbstractCustomAuthenticationToken<?>> authenticationTokenMap = new HashMap<>();

    public CustomGrantAuthenticationFilter() {
        this(DEFAULT_FILTER_PROCESS_URI);
    }

    public CustomGrantAuthenticationFilter(String defaultFilterProcessesUrl) {
        super(new AntPathRequestMatcher(defaultFilterProcessesUrl, HttpMethod.POST.name()));
    }

    public <T extends AbstractCustomAuthenticationToken> void addAuthenticationTokenConvert(T authenticationToken) {
        LoginType loginType = authenticationToken.loginType();
        Assert.notNull(loginType, authenticationToken.getClass().getName() + "的loginType为空");

        authenticationTokenMap.put(loginType.getType(), authenticationToken);
    }

    @Override
    public Authentication attemptAuthentication(HttpServletRequest request, HttpServletResponse response) throws AuthenticationException, IOException, ServletException {
        // 判断登录类型
        String loginType = request.getParameter(CustomOAuth2ParameterNames.LOGIN_TYPE);
        if (!StringUtils.hasText(loginType)) {
            throw new AuthenticationServiceException("未知登录类型");
        }

        // 转token
        AbstractCustomAuthenticationToken<?> authenticationToken = authenticationTokenMap.get(loginType);
        if (authenticationToken == null) {
            throw new AuthenticationServiceException("不支持的登录类型");
        }
        AbstractCustomAuthenticationToken<?> authentication = authenticationToken.convert(request);
        setDetails(request, authentication);

        // 开始认证
        return this.getAuthenticationManager().authenticate(authentication);
    }

    private void setDetails(HttpServletRequest request, AbstractAuthenticationToken authentication) {
        authentication.setDetails(this.authenticationDetailsSource.buildDetails(request));
    }
}
