package com.elitesland.cloudt.authorization.api.provider.security.grant.password;

import com.elitesland.cloudt.authorization.api.client.util.HttpServletUtil;
import com.elitesland.cloudt.authorization.api.provider.common.CustomOAuth2ParameterNames;
import com.elitesland.cloudt.authorization.api.provider.common.LoginType;
import com.elitesland.cloudt.authorization.api.provider.security.grant.AbstractCustomAuthenticationToken;
import com.elitesland.yst.common.constant.Terminal;
import org.springframework.lang.NonNull;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.core.endpoint.OAuth2ParameterNames;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Collection;

/**
 * 用户名与密码的身份认证令牌.
 *
 * @author Kaiser（wang shao）
 * @date 2022/01/01
 */
public class UsernamePasswordAuthenticationToken extends AbstractCustomAuthenticationToken<UsernamePasswordAuthenticationToken> {
    private static final long serialVersionUID = 3291894847806095884L;

    /**
     * 验证码内容
     */
    private String captchaText;
    /**
     * 验证码key
     */
    private String captchaKey;
    /**
     * 是否必须验证码
     */
    private Boolean captchaNeeded;

    public UsernamePasswordAuthenticationToken() {
        super(null, null);
    }

    public UsernamePasswordAuthenticationToken(Object principal, Object credentials) {
        super(principal, credentials);
    }

    public UsernamePasswordAuthenticationToken(Object principal, Object credentials, Collection<? extends GrantedAuthority> authorities) {
        super(principal, credentials, authorities);
    }

    @NonNull
    @Override
    public LoginType loginType() {
        return LoginType.PASSWORD;
    }

    @NonNull
    @Override
    public UsernamePasswordAuthenticationToken convert(@NonNull HttpServletRequest request) {
        MultiValueMap<String, String> parameters = HttpServletUtil.getParameters(request);
        String terminalParam = parameters.getFirst(CustomOAuth2ParameterNames.TERMINAL);
        Terminal terminal = null;
        if (StringUtils.hasText(terminalParam)) {
            terminal = Terminal.parse(terminalParam);
        }

        String captchaNeededStr = parameters.getFirst(CustomOAuth2ParameterNames.CAPTCHA_NEED);
        // 为空或者true时必须有验证码
        Boolean captchaNeeded = !StringUtils.hasText(captchaNeededStr) || Boolean.parseBoolean(captchaNeededStr);

        UsernamePasswordAuthenticationToken authenticationToken = new UsernamePasswordAuthenticationToken();
        authenticationToken.setTerminal(terminal);
        authenticationToken.setPrincipal(parameters.getFirst(OAuth2ParameterNames.USERNAME));
        authenticationToken.setCredentials(parameters.getFirst(OAuth2ParameterNames.PASSWORD));
        authenticationToken.setAuthenticated(false);

        authenticationToken.setCaptchaText(parameters.getFirst(CustomOAuth2ParameterNames.CAPTCHA_TEXT));
        authenticationToken.setCaptchaKey(parameters.getFirst(CustomOAuth2ParameterNames.CAPTCHA_UID));
        authenticationToken.setCaptchaNeeded(captchaNeeded);

        return authenticationToken;
    }

    public String getCaptchaText() {
        return captchaText;
    }

    public String getCaptchaKey() {
        return captchaKey;
    }

    public Boolean getCaptchaNeeded() {
        return captchaNeeded;
    }

    public void setCaptchaText(String captchaText) {
        this.captchaText = captchaText;
    }

    public void setCaptchaKey(String captchaKey) {
        this.captchaKey = captchaKey;
    }

    public void setCaptchaNeeded(Boolean captchaNeeded) {
        this.captchaNeeded = captchaNeeded;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj);
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }
}
