package com.elitesland.cloudt.authorization.api.provider.security.handler;

import com.elitesland.cloudt.authorization.api.client.config.support.AuthenticationCache;
import com.elitesland.cloudt.authorization.api.client.model.OAuthToken;
import com.elitesland.cloudt.authorization.api.client.config.AuthorizationProperties;
import com.elitesland.cloudt.authorization.api.client.config.security.handler.DefaultAuthenticationSuccessHandler;
import org.springframework.lang.NonNull;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.OAuth2AccessToken;
import org.springframework.security.oauth2.core.OAuth2TokenType;
import org.springframework.security.oauth2.jose.jws.SignatureAlgorithm;
import org.springframework.security.oauth2.jwt.*;
import org.springframework.security.oauth2.server.authorization.token.JwtEncodingContext;
import org.springframework.security.oauth2.server.authorization.token.OAuth2TokenCustomizer;

import javax.servlet.http.HttpServletRequest;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;

/**
 * JWT认证成功处理器.
 * <p>
 * 认证成功后返回jwt
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/1
 */
public class JwtAuthenticationSuccessHandler extends DefaultAuthenticationSuccessHandler {
    private static final String ISSUER_DEFAULT = "cloudt";

    private final JwtEncoder jwtEncoder;
    private OAuth2TokenCustomizer<JwtEncodingContext> tokenCustomizer;

    public JwtAuthenticationSuccessHandler(AuthorizationProperties authorizationProperties, AuthenticationCache authenticationCache, @NonNull JwtEncoder jwtEncoder) {
        super(authorizationProperties, authenticationCache);
        this.jwtEncoder = jwtEncoder;
    }

    @Override
    protected Object convertResponseResult(HttpServletRequest request, Authentication authentication) {
        Jwt jwt = generateJwt(authentication);
        return convertToken(jwt);
    }

    @Override
    protected String generateCacheKey(HttpServletRequest request, Object responseResult, Authentication authentication) {
        OAuthToken authToken = (OAuthToken) responseResult;
        return authToken.getAccessToken();
    }

    /**
     * 生成jwt
     *
     * @param authentication authentication
     * @return jwt
     */
    private Jwt generateJwt(Authentication authentication) {
        Instant issuedAt = Instant.now();

        // claimBuilder
        var claimBuilder = JwtClaimsSet.builder()
                .issuer(ISSUER_DEFAULT)
                .subject(authentication.getName())
                .audience(Collections.singletonList(authentication.getName()))
                .issuedAt(issuedAt)
                // 目前默认30分钟，后期可配置化
                .expiresAt(issuedAt.plus(30, ChronoUnit.MINUTES));

        JwsHeader.Builder headersBuilder = JwsHeader.with(SignatureAlgorithm.RS256);
        JwtEncodingContext.Builder jwtContextBuilder = JwtEncodingContext.with(headersBuilder, claimBuilder)
                .principal(authentication)
                .tokenType(OAuth2TokenType.ACCESS_TOKEN)
                .authorizationGrantType(AuthorizationGrantType.JWT_BEARER);

        JwtEncodingContext jwtContext = jwtContextBuilder.build();
        // 自定义claim
        if (tokenCustomizer != null) {
            tokenCustomizer.customize(jwtContext);
        }

        JwsHeader headers = headersBuilder.build();
        JwtClaimsSet claims = claimBuilder.build();

        return jwtEncoder.encode(JwtEncoderParameters.from(headers, claims));
    }

    public void setTokenCustomizer(OAuth2TokenCustomizer<JwtEncodingContext> tokenCustomizer) {
        this.tokenCustomizer = tokenCustomizer;
    }

    /**
     * jwt转自定义token格式
     *
     * @param jwt jwt
     * @return token
     */
    private OAuthToken convertToken(Jwt jwt) {
        OAuthToken token = new OAuthToken();
        token.setAccessToken(jwt.getTokenValue());
        // 目前使用bearer
        token.setTokenType(OAuth2AccessToken.TokenType.BEARER.getValue());

        long expiresIn = ChronoUnit.SECONDS.between(Instant.now(), jwt.getExpiresAt());
        token.setExpiresIn(expiresIn);
        token.setScope(Collections.emptySet());

        // 暂时不用refreshToken
        token.setRefreshToken(null);

        return token;
    }
}
