package com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server;

import com.elitesland.cloudt.authorization.api.provider.security.handler.oauth2.server.support.OAuth2AuthorizationCodeRequestCache;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.base.ApiResult;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.server.authorization.authentication.OAuth2AuthorizationCodeRequestAuthenticationToken;
import org.springframework.security.oauth2.server.authorization.web.authentication.OAuth2AuthorizationCodeRequestAuthenticationConverter;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.OrRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.Duration;

/**
 * 未认证时的处理.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/3
 */
@Log4j2
public class OAuth2ServerAuthenticationEntryPointHandler extends AbstractOAuth2ServerHandler implements AuthenticationEntryPoint {
    private static final Duration CODE_REQUEST_TTL = Duration.ofMinutes(5);

    private final OAuth2AuthorizationCodeRequestCache authorizationCodeRequestCache;
    private final RequestMatcher oauth2AuthorizationEndpointRequestMatcher;


    public OAuth2ServerAuthenticationEntryPointHandler(OAuth2AuthorizationCodeRequestCache authorizationCodeRequestCache, String oauth2AuthorizationEndpoint) {
        this.authorizationCodeRequestCache = authorizationCodeRequestCache;
        this.oauth2AuthorizationEndpointRequestMatcher = buildOAuth2AuthorizationEndpointRequestMatcher(oauth2AuthorizationEndpoint);
    }

    @Override
    public void commence(HttpServletRequest request, HttpServletResponse response, AuthenticationException authException) throws IOException, ServletException {
        log.info("未认证请求：{}", request.getRequestURI());
        cacheAuthorizeRequest(request);

        writeResponse(response, ApiResult.fail(ApiCode.UNAUTHORIZED, "未认证或身份认证已过期，请重新登录"), HttpStatus.UNAUTHORIZED);
    }

    private void cacheAuthorizeRequest(HttpServletRequest request) {
        if (!oauth2AuthorizationEndpointRequestMatcher.matches(request)) {
            return;
        }

        OAuth2AuthorizationCodeRequestAuthenticationToken codeRequest = (OAuth2AuthorizationCodeRequestAuthenticationToken) new OAuth2AuthorizationCodeRequestAuthenticationConverter().convert(request);
        authorizationCodeRequestCache.setAuthenticationToken(codeRequest.getState(), codeRequest, CODE_REQUEST_TTL);
    }

    private RequestMatcher buildOAuth2AuthorizationEndpointRequestMatcher(String oauth2Authorizationendpoint) {
        return new OrRequestMatcher(
                new AntPathRequestMatcher(
                        oauth2Authorizationendpoint,
                        HttpMethod.GET.name()),
                new AntPathRequestMatcher(
                        oauth2Authorizationendpoint,
                        HttpMethod.POST.name()));
    }
}
