package com.elitesland.cloudt.authorization.api.provider.security.impl;

import com.elitesland.cloudt.authorization.api.client.common.AuthorizationException;
import com.elitesland.cloudt.authorization.api.provider.config.system.TenantProperties;
import com.elitesland.cloudt.authorization.api.provider.security.AuthenticationCheckService;
import com.elitesland.cloudt.authorization.api.provider.security.grant.AbstractCustomAuthenticationToken;
import com.elitesland.cloudt.context.util.HttpServletUtil;
import com.elitesland.yst.common.constant.Terminal;
import com.elitesland.yst.core.provider.tenant.TenantClientProvider;
import com.elitesland.yst.security.entity.GeneralUserDetails;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.AuthenticationException;
import org.springframework.util.CollectionUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * 用户的租户信息检查.
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/20
 */
@Log4j2
public class TenantAuthenticationCheckServiceImpl<T extends AbstractCustomAuthenticationToken<T>> implements AuthenticationCheckService<T> {

    private final TenantProperties tenantProperties;
    private final TenantClientProvider tenantClientProvider;

    public TenantAuthenticationCheckServiceImpl(TenantProperties tenantProperties, TenantClientProvider tenantClientProvider) {
        this.tenantProperties = tenantProperties;
        this.tenantClientProvider = tenantClientProvider;
    }

    @Override
    public void additionalAuthenticationChecks(GeneralUserDetails user, T authentication) throws AuthenticationException {
        if (!tenantClientProvider.enabledTenant() || !Boolean.TRUE.equals(tenantProperties.isLimitTenantLogin())) {
            // 未启用租户   未限制租户的登录
            return;
        }

        if (CollectionUtils.isEmpty(user.getUser().getSysTenantDTOList())) {
            // 无租户用户
            return;
        }

        if (Terminal.BACKEND.name().equals(authentication.getTerminal())) {
            // 默认站点下只有运营可以登录
            if (user.isSystemAdmin() || user.isOperation()) {
                return;
            }

            if (tenantClientProvider.isDefaultDomainRequest()) {
                throw new AuthorizationException("请转至对应租户下的站点登录");
            }

            // 判断租户与当前登录用户所属的租户是否一致
            var tenant = tenantClientProvider.obtainTenantFromRequest();
            if (tenant == null) {
                log.info("请求域名：{}", obtainDomain());
                throw new AuthorizationException("未知站点所属租户");
            }

            if (user.getTenant() == null || !user.getTenant().getId().equals(tenant.getId())) {
                log.warn("登录用户的租户：{}", user.getTenant());
                throw new AuthorizationException("请转至租户下的站点登录");
            }
        }
    }

    private String obtainDomain() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if (requestAttributes == null) {
            return null;
        }

        HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();
        String uri = request.getRequestURL().toString();
        String domain = HttpServletUtil.obtainDomain(request);
        log.info("请求路径：{}, {}", uri, domain);
        return domain;
    }
}
