package com.elitesland.cloudt.authorization.api.provider.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.IdUtil;
import com.elitesland.cloudt.authorization.api.provider.model.entity.OAuth2AuthorizationConsentDO;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2AuthorizationConsentRepo;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2AuthorizationConsentRepoProc;
import com.elitesland.yst.core.annotation.TenantTransaction;
import com.elitesland.yst.core.annotation.common.TenantIsolateType;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.oauth2.server.authorization.OAuth2AuthorizationConsent;
import org.springframework.security.oauth2.server.authorization.OAuth2AuthorizationConsentService;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClient;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClientRepository;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/9
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
public class JpaOAuth2AuthorizationConsentService extends BaseCustomAuthorizationService implements OAuth2AuthorizationConsentService {

    private final RegisteredClientRepository registeredClientRepository;
    private final OAuth2AuthorizationConsentRepo oAuth2AuthorizationConsentRepo;
    private final OAuth2AuthorizationConsentRepoProc oAuth2AuthorizationConsentRepoProc;

    public JpaOAuth2AuthorizationConsentService(RegisteredClientRepository registeredClientRepository,
                                                OAuth2AuthorizationConsentRepo oAuth2AuthorizationConsentRepo,
                                                OAuth2AuthorizationConsentRepoProc oAuth2AuthorizationConsentRepoProc) {
        this.registeredClientRepository = registeredClientRepository;
        this.oAuth2AuthorizationConsentRepo = oAuth2AuthorizationConsentRepo;
        this.oAuth2AuthorizationConsentRepoProc = oAuth2AuthorizationConsentRepoProc;
    }

    @Override
    public void save(OAuth2AuthorizationConsent authorizationConsent) {
        if (authorizationConsent == null) {
            log.error("保存内容为空");
            return;
        }
        Assert.hasText(authorizationConsent.getPrincipalName(), "principal为空");
        Assert.hasText(authorizationConsent.getRegisteredClientId(), "clientId为空");

        OAuth2AuthorizationConsentDO consentDO = toDo(authorizationConsent);
        oAuth2AuthorizationConsentRepo.save(consentDO);
    }

    @Override
    public void remove(OAuth2AuthorizationConsent authorizationConsent) {
        Assert.hasText(authorizationConsent.getPrincipalName(), "principal为空");
        Assert.hasText(authorizationConsent.getRegisteredClientId(), "clientId为空");

        oAuth2AuthorizationConsentRepoProc.delete(authorizationConsent.getRegisteredClientId(), authorizationConsent.getPrincipalName());
    }

    @Override
    public OAuth2AuthorizationConsent findById(String registeredClientId, String principalName) {
        Assert.hasText(principalName, "principal为空");
        Assert.hasText(registeredClientId, "clientId为空");

        OAuth2AuthorizationConsentDO consentDO = oAuth2AuthorizationConsentRepoProc.get(registeredClientId, principalName);
        return toBo(consentDO);
    }

    private OAuth2AuthorizationConsentDO toDo(OAuth2AuthorizationConsent authorizationConsent) {
        OAuth2AuthorizationConsentDO consentDO = new OAuth2AuthorizationConsentDO();

        String id = oAuth2AuthorizationConsentRepoProc.getId(authorizationConsent.getRegisteredClientId(), authorizationConsent.getPrincipalName());
        consentDO.setId(CharSequenceUtil.blankToDefault(id, IdUtil.fastSimpleUUID()));
        consentDO.setRegisteredClientId(authorizationConsent.getRegisteredClientId());
        consentDO.setPrincipalName(authorizationConsent.getPrincipalName());

        if (!CollectionUtils.isEmpty(authorizationConsent.getAuthorities())) {
            Set<String> authorities = authorizationConsent.getAuthorities().stream().map(GrantedAuthority::getAuthority).collect(Collectors.toSet());
            consentDO.setAuthorities(toJsonString(authorities));
        }

        return consentDO;
    }

    private OAuth2AuthorizationConsent toBo(OAuth2AuthorizationConsentDO authorizationConsentDo) {
        if (authorizationConsentDo == null) {
            return null;
        }

        Assert.hasText(authorizationConsentDo.getRegisteredClientId(), "clientId为空");
        RegisteredClient client = registeredClientRepository.findById(authorizationConsentDo.getRegisteredClientId());
        Assert.notNull(client, "client不存在");

        OAuth2AuthorizationConsent.Builder builder = OAuth2AuthorizationConsent.withId(authorizationConsentDo.getRegisteredClientId(),
                authorizationConsentDo.getPrincipalName());

        if (StringUtils.hasText(authorizationConsentDo.getAuthorities())) {
            Set<String> authorities = toJsonObj(authorizationConsentDo.getAuthorities(), new TypeReference<>() {
            });
            for (String authority : authorities) {
                builder.authority(new SimpleGrantedAuthority(authority));
            }
        }
        return builder.build();
    }
}
