package com.elitesland.cloudt.authorization.api.provider.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cloudt.authorization.api.provider.model.entity.OAuth2AuthorizationDO;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2AuthenticationRepo;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2AuthenticationRepoProc;
import com.elitesland.yst.core.annotation.TenantTransaction;
import com.elitesland.yst.core.annotation.common.TenantIsolateType;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.oauth2.core.*;
import org.springframework.security.oauth2.core.endpoint.OAuth2ParameterNames;
import org.springframework.security.oauth2.core.oidc.OidcIdToken;
import org.springframework.security.oauth2.server.authorization.OAuth2Authorization;
import org.springframework.security.oauth2.server.authorization.OAuth2AuthorizationService;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClient;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClientRepository;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/9
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
public class JpaOAuth2AuthorizationService extends BaseCustomAuthorizationService implements OAuth2AuthorizationService {

    private final RegisteredClientRepository registeredClientRepository;
    private final OAuth2AuthenticationRepo oAuth2AuthenticationRepo;
    private final OAuth2AuthenticationRepoProc oAuth2AuthenticationRepoProc;

    public JpaOAuth2AuthorizationService(RegisteredClientRepository registeredClientRepository,
                                         OAuth2AuthenticationRepo oAuth2AuthenticationRepo,
                                         OAuth2AuthenticationRepoProc oAuth2AuthenticationRepoProc) {
        this.registeredClientRepository = registeredClientRepository;
        this.oAuth2AuthenticationRepo = oAuth2AuthenticationRepo;
        this.oAuth2AuthenticationRepoProc = oAuth2AuthenticationRepoProc;
    }

    @Override
    public void save(OAuth2Authorization authorization) {
        if (authorization == null) {
            log.error("authorization为空");
            return;
        }

        OAuth2AuthorizationDO auth2AuthorizationDO = toDo(authorization);
        oAuth2AuthenticationRepo.save(auth2AuthorizationDO);
    }

    @Override
    public void remove(OAuth2Authorization authorization) {
        if (authorization == null || !StringUtils.hasText(authorization.getId())) {
            log.error("删除authorization失败");
            return;
        }

        oAuth2AuthenticationRepoProc.delete(authorization.getId());
    }

    @Override
    public OAuth2Authorization findById(String id) {
        if (!StringUtils.hasText(id)) {
            log.error("查询authorization失败，ID为空");
            return null;
        }

        OAuth2AuthorizationDO authorizationDO = oAuth2AuthenticationRepo.findById(id).orElse(null);
        return toBo(authorizationDO);
    }

    @Override
    public OAuth2Authorization findByToken(String token, OAuth2TokenType tokenType) {
        Assert.hasText(token, "token为空");

        OAuth2AuthorizationDO authorizationDO = oAuth2AuthenticationRepoProc.getByToken(token, tokenType);
        return toBo(authorizationDO);
    }

    private OAuth2AuthorizationDO toDo(OAuth2Authorization authorization) {
        OAuth2AuthorizationDO authorizationDO = new OAuth2AuthorizationDO();

        authorizationDO.setId(CharSequenceUtil.blankToDefault(authorization.getId(), IdUtil.fastSimpleUUID()));
        authorizationDO.setRegisteredClientId(authorization.getRegisteredClientId());
        authorizationDO.setPrincipalName(authorization.getPrincipalName());
        if (authorization.getAuthorizationGrantType() != null) {
            authorizationDO.setAuthorizationGrantType(authorization.getAuthorizationGrantType().getValue());
        }
        authorizationDO.setAttributes(toJsonString(authorization.getAttributes()));

        authorizationDO.setState(authorization.getAttribute(OAuth2ParameterNames.STATE));

        OAuth2Authorization.Token<OAuth2AuthorizationCode> authorizationCode =
                authorization.getToken(OAuth2AuthorizationCode.class);
        if (authorizationCode != null) {
            OAuth2AuthorizationCode token = authorizationCode.getToken();
            if (token != null) {
                authorizationDO.setAuthorizationCodeValue(token.getTokenValue());
                authorizationDO.setAuthorizationCodeIssuedAt(toLocalDateTime(token.getIssuedAt()));
                authorizationDO.setAuthorizationCodeExpiresAt(toLocalDateTime(token.getExpiresAt()));
            }
            authorizationDO.setAuthorizationCodeMetadata(toJsonString(authorizationCode.getMetadata()));
        }

        OAuth2Authorization.Token<OAuth2AccessToken> accessToken =
                authorization.getToken(OAuth2AccessToken.class);
        if (accessToken != null) {
            OAuth2AccessToken token = accessToken.getToken();
            if (token != null) {
                authorizationDO.setAccessTokenValue(token.getTokenValue());
                authorizationDO.setAccessTokenIssuedAt(toLocalDateTime(token.getIssuedAt()));
                authorizationDO.setAccessTokenExpiresAt(toLocalDateTime(token.getExpiresAt()));
                if (token.getTokenType() != null) {
                    authorizationDO.setAccessTokenType(token.getTokenType().getValue());
                }
                authorizationDO.setAccessTokenScopes(toJsonString(token.getScopes()));
            }
            authorizationDO.setAccessTokenMetadata(toJsonString(accessToken.getMetadata()));
        }

        OAuth2Authorization.Token<OidcIdToken> oidcIdToken = authorization.getToken(OidcIdToken.class);
        if (oidcIdToken != null) {
            OidcIdToken token = oidcIdToken.getToken();
            if (token != null) {
                authorizationDO.setOidcIdTokenValue(token.getTokenValue());
                authorizationDO.setOidcIdTokenIssuedAt(toLocalDateTime(token.getIssuedAt()));
                authorizationDO.setOidcIdTokenExpiresAt(toLocalDateTime(token.getExpiresAt()));
            }
            authorizationDO.setOidcIdTokenMetadata(toJsonString(oidcIdToken.getMetadata()));
        }

        OAuth2Authorization.Token<OAuth2RefreshToken> refreshToken = authorization.getRefreshToken();
        if (refreshToken != null) {
            OAuth2RefreshToken token = refreshToken.getToken();
            if (token != null) {
                authorizationDO.setRefreshTokenValue(token.getTokenValue());
                authorizationDO.setRefreshTokenIssuedAt(toLocalDateTime(token.getIssuedAt()));
                authorizationDO.setRefreshTokenExpiresAt(toLocalDateTime(token.getExpiresAt()));
            }
            authorizationDO.setRefreshTokenMetadata(toJsonString(refreshToken.getMetadata()));
        }

        return authorizationDO;
    }

    public OAuth2Authorization toBo(OAuth2AuthorizationDO authorizationDO) {
        if (authorizationDO == null) {
            return null;
        }

        String clientId = authorizationDO.getRegisteredClientId();
        Assert.hasText(clientId, "未知token的clientId");

        RegisteredClient client = registeredClientRepository.findById(clientId);
        Assert.notNull(client, "客户端不存在");

        OAuth2Authorization.Builder builder = OAuth2Authorization.withRegisteredClient(client);
        builder.id(authorizationDO.getId());
        builder.principalName(authorizationDO.getPrincipalName());
        if (StringUtils.hasText(authorizationDO.getAuthorizationGrantType())) {
            builder.authorizationGrantType(new AuthorizationGrantType(authorizationDO.getAuthorizationGrantType()));
        }
        Map<String, Object> attributes = toJsonObj(authorizationDO.getAttributes(), new TypeReference<>() {
        });
        if (!CollectionUtils.isEmpty(attributes)) {
            builder.attributes(t -> t.putAll(attributes));

            if (StringUtils.hasText(authorizationDO.getState())) {
                builder.attribute(OAuth2ParameterNames.STATE, authorizationDO.getState());
            }
        }

        if (ArrayUtil.isNotEmpty(authorizationDO.getAuthorizationCodeValue())) {
            String authorizationCodeValue = authorizationDO.getAuthorizationCodeValue();
            Map<String, Object> authorizationCodeMetadata = toJsonObj(authorizationDO.getAuthorizationCodeMetadata(), new TypeReference<>() {
            });

            OAuth2AuthorizationCode authorizationCode = new OAuth2AuthorizationCode(
                    authorizationCodeValue, toInstant(authorizationDO.getAuthorizationCodeIssuedAt()),
                    toInstant(authorizationDO.getAuthorizationCodeExpiresAt()));
            builder.token(authorizationCode, metadata -> metadata.putAll(ObjectUtil.defaultIfNull(authorizationCodeMetadata, Collections.emptyMap())));
        }

        if (ArrayUtil.isNotEmpty(authorizationDO.getAccessTokenValue())) {
            String accessTokenValue = authorizationDO.getAccessTokenValue();
            Map<String, Object> accessTokenMetadata = toJsonObj(authorizationDO.getAccessTokenMetadata(), new TypeReference<>() {
            });
            OAuth2AccessToken.TokenType tokenType = null;
            if (OAuth2AccessToken.TokenType.BEARER.getValue().equalsIgnoreCase(authorizationDO.getAccessTokenType())) {
                tokenType = OAuth2AccessToken.TokenType.BEARER;
            }

            Set<String> scopes = toJsonObj(authorizationDO.getAccessTokenScopes(), new TypeReference<>() {
            });
            OAuth2AccessToken accessToken = new OAuth2AccessToken(tokenType, accessTokenValue,
                    toInstant(authorizationDO.getAccessTokenIssuedAt()),
                    toInstant(authorizationDO.getAccessTokenExpiresAt()), scopes);
            builder.token(accessToken, metadata -> metadata.putAll(ObjectUtil.defaultIfNull(accessTokenMetadata, Collections.emptyMap())));
        }

        if (ArrayUtil.isNotEmpty(authorizationDO.getOidcIdTokenValue())) {
            String oidcIdTokenValue = authorizationDO.getOidcIdTokenValue();
            Map<String, Object> oidcTokenMetadata = toJsonObj(authorizationDO.getOidcIdTokenMetadata(), new TypeReference<>() {
            });
            Map<String, Object> tokenMetadata = oidcTokenMetadata == null ? null : (Map<String, Object>) oidcTokenMetadata.get(OAuth2Authorization.Token.CLAIMS_METADATA_NAME);

            OidcIdToken oidcToken = new OidcIdToken(
                    oidcIdTokenValue, toInstant(authorizationDO.getOidcIdTokenIssuedAt()),
                    toInstant(authorizationDO.getOidcIdTokenExpiresAt()), tokenMetadata);
            builder.token(oidcToken, t -> t.putAll(ObjectUtil.defaultIfNull(oidcTokenMetadata, Collections.emptyMap())));
        }

        if (ArrayUtil.isNotEmpty(authorizationDO.getRefreshTokenValue())) {
            String refreshTokenValue = authorizationDO.getRefreshTokenValue();
            Map<String, Object> refreshTokenMetadata = toJsonObj(authorizationDO.getRefreshTokenMetadata(), new TypeReference<>() {
            });

            OAuth2RefreshToken refreshToken = new OAuth2RefreshToken(
                    refreshTokenValue, toInstant(authorizationDO.getRefreshTokenIssuedAt()),
                    toInstant(authorizationDO.getRefreshTokenExpiresAt()));
            builder.token(refreshToken, metadata -> metadata.putAll(ObjectUtil.defaultIfNull(refreshTokenMetadata, Collections.emptyMap())));
        }

        return builder.build();
    }
}
