package com.elitesland.cloudt.authorization.api.provider.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.IdUtil;
import com.elitesland.cloudt.authorization.api.provider.model.entity.OAuth2RegisteredClientDO;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2RegisteredClientRepo;
import com.elitesland.cloudt.authorization.api.provider.service.reposotory.OAuth2RegisteredClientRepoProc;
import com.elitesland.yst.core.annotation.TenantTransaction;
import com.elitesland.yst.core.annotation.common.TenantIsolateType;
import com.fasterxml.jackson.core.type.TypeReference;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.log4j.Log4j2;
import org.springframework.security.oauth2.core.AuthorizationGrantType;
import org.springframework.security.oauth2.core.ClientAuthenticationMethod;
import org.springframework.security.oauth2.core.OAuth2TokenFormat;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClient;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClientRepository;
import org.springframework.security.oauth2.server.authorization.config.ClientSettings;
import org.springframework.security.oauth2.server.authorization.config.ConfigurationSettingNames;
import org.springframework.security.oauth2.server.authorization.config.TokenSettings;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2022/6/9
 */
@Log4j2
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
public class JpaRegisteredClientRepository extends BaseCustomAuthorizationService implements RegisteredClientRepository {

    private final OAuth2RegisteredClientRepo oAuth2RegisteredClientRepo;
    private final OAuth2RegisteredClientRepoProc oAuth2RegisteredClientRepoProc;
    private final Cache<String, RegisteredClient> clientCacheForId = Caffeine.newBuilder()
            .maximumSize(50)
            .expireAfterWrite(Duration.ofHours(2))
            .build();
    private final Cache<String, RegisteredClient> clientCacheForClientId = Caffeine.newBuilder()
            .maximumSize(50)
            .expireAfterWrite(Duration.ofHours(2))
            .build();

    public JpaRegisteredClientRepository(OAuth2RegisteredClientRepo oAuth2RegisteredClientRepo,
                                         OAuth2RegisteredClientRepoProc oAuth2RegisteredClientRepoProc) {
        this.oAuth2RegisteredClientRepo = oAuth2RegisteredClientRepo;
        this.oAuth2RegisteredClientRepoProc = oAuth2RegisteredClientRepoProc;
    }

    @Override
    public void save(RegisteredClient registeredClient) {
        if (registeredClient == null) {
            log.error("保存RegisteredClient为空");
            return;
        }

        Assert.hasText(registeredClient.getClientId(), "clientId为空");
        boolean exists = oAuth2RegisteredClientRepoProc.existsClientId(registeredClient.getClientId(), registeredClient.getId());
        Assert.isTrue(!exists, "clientId已存在");

        OAuth2RegisteredClientDO registeredClientDO = toDo(registeredClient);
        oAuth2RegisteredClientRepo.save(registeredClientDO);
    }

    @Override
    public RegisteredClient findById(String id) {
        if (!StringUtils.hasText(id)) {
            log.error("ID为空");
            return null;
        }

        // 先从缓存中获取
        RegisteredClient client = clientCacheForId.getIfPresent(id);
        if (client != null) {
            return client;
        }

        // 从数据库查询
        OAuth2RegisteredClientDO clientDO = oAuth2RegisteredClientRepo.findById(id).orElse(null);
        if (clientDO == null) {
            log.warn("RegisteredClient【{}】不存在", id);
            return null;
        }

        client = toBo(clientDO);
        clientCacheForId.put(id, client);

        return client;
    }

    @Override
    public RegisteredClient findByClientId(String clientId) {
        if (!StringUtils.hasText(clientId)) {
            log.error("clientId为空");
            return null;
        }

        // 先从缓存中获取
        RegisteredClient client = clientCacheForClientId.getIfPresent(clientId);
        if (client != null) {
            return client;
        }

        OAuth2RegisteredClientDO clientDO = oAuth2RegisteredClientRepoProc.getByClientId(clientId);
        if (clientDO == null) {
            log.warn("RegisteredClient【{}】不存在", clientId);
            return null;
        }

        client = toBo(clientDO);
        clientCacheForClientId.put(clientId, client);

        return client;
    }

    private OAuth2RegisteredClientDO toDo(RegisteredClient registeredClient) {
        OAuth2RegisteredClientDO clientDO = new OAuth2RegisteredClientDO();
        clientDO.setId(CharSequenceUtil.blankToDefault(registeredClient.getId(), IdUtil.fastSimpleUUID()));
        clientDO.setClientId(registeredClient.getClientId());
        clientDO.setClientIdIssuedAt(toLocalDateTime(registeredClient.getClientIdIssuedAt()));
        clientDO.setClientSecret(registeredClient.getClientSecret());
        clientDO.setClientSecretExpiresAt(toLocalDateTime(registeredClient.getClientSecretExpiresAt()));
        clientDO.setClientName(registeredClient.getClientName());

        if (!CollectionUtils.isEmpty(registeredClient.getClientAuthenticationMethods())) {
            String methods = registeredClient.getClientAuthenticationMethods().stream()
                    .map(ClientAuthenticationMethod::getValue)
                    .collect(Collectors.joining(","));
            clientDO.setClientAuthenticationMethods(methods);
        }

        if (!CollectionUtils.isEmpty(registeredClient.getAuthorizationGrantTypes())) {
            String grantTypes = registeredClient.getAuthorizationGrantTypes().stream()
                    .map(AuthorizationGrantType::getValue)
                    .collect(Collectors.joining(","));
            clientDO.setAuthorizationGrantTypes(grantTypes);
        }

        clientDO.setRedirectUris(StringUtils.collectionToCommaDelimitedString(registeredClient.getRedirectUris()));
        clientDO.setScopes(StringUtils.collectionToCommaDelimitedString(registeredClient.getScopes()));
        clientDO.setClientSettings(toJsonString(registeredClient.getClientSettings().getSettings()));
        clientDO.setTokenSettings(toJsonString(registeredClient.getTokenSettings().getSettings()));

        return clientDO;
    }

    private RegisteredClient toBo(OAuth2RegisteredClientDO clientDO) {
        Set<String> clientAuthenticationMethods = StringUtils.commaDelimitedListToSet(clientDO.getClientAuthenticationMethods());
        Set<String> authorizationGrantTypes = StringUtils.commaDelimitedListToSet(clientDO.getAuthorizationGrantTypes());
        Set<String> redirectUris = StringUtils.commaDelimitedListToSet(clientDO.getRedirectUris());
        Set<String> clientScopes = StringUtils.commaDelimitedListToSet(clientDO.getScopes());

        RegisteredClient.Builder builder = RegisteredClient.withId(clientDO.getId())
                .clientId(clientDO.getClientId())
                .clientIdIssuedAt(toInstant(clientDO.getClientIdIssuedAt()))
                .clientSecret(clientDO.getClientSecret())
                .clientSecretExpiresAt(toInstant(clientDO.getClientSecretExpiresAt()))
                .clientName(clientDO.getClientName())
                .clientAuthenticationMethods(authenticationMethods ->
                        clientAuthenticationMethods.forEach(authenticationMethod ->
                                authenticationMethods.add(new ClientAuthenticationMethod(authenticationMethod))))
                .authorizationGrantTypes(grantTypes ->
                        authorizationGrantTypes.forEach(grantType ->
                                grantTypes.add(new AuthorizationGrantType(grantType))))
                .redirectUris(uris -> uris.addAll(redirectUris))
                .scopes(scopes -> scopes.addAll(clientScopes));

        // client setting
        Map<String, Object> clientSettingsMap = toJsonObj(clientDO.getClientSettings(), new TypeReference<>() {
        });
        builder.clientSettings(ClientSettings.withSettings(clientSettingsMap).build());

        // token setting
        Map<String, Object> tokenSettingsMap = toJsonObj(clientDO.getTokenSettings(), new TypeReference<>() {
        });
        TokenSettings.Builder tokenSettingsBuilder = TokenSettings.withSettings(tokenSettingsMap);
        if (!tokenSettingsMap.containsKey(ConfigurationSettingNames.Token.ACCESS_TOKEN_FORMAT)) {
            tokenSettingsBuilder.accessTokenFormat(OAuth2TokenFormat.SELF_CONTAINED);
        }

        builder.tokenSettings(tokenSettingsBuilder.build());

        return builder.build();
    }
}
