package com.elitesland.cloudt.authorization.api.provider.web.controller;

import com.elitesland.cloudt.authorization.api.client.model.OAuthToken;
import com.elitesland.cloudt.authorization.api.provider.model.vo.resp.OAuth2SettingRespVO;
import com.elitesland.cloudt.authorization.api.provider.provider.oauth2.client.OAuth2ClientProvider;
import com.elitesland.yst.common.base.ApiResult;
import org.springframework.data.repository.query.Param;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.validation.constraints.NotBlank;

/**
 * OAuth2登录相关扩展接口.
 *
 * @author Kaiser（wang shao）
 * @date 2022/7/5
 */
@ResponseBody
@RequestMapping(value = "/oauth2/login", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class OAuth2LoginSupportController {

    private final OAuth2ClientProvider oAuth2ClientProvider;

    public OAuth2LoginSupportController(OAuth2ClientProvider oAuth2ClientProvider) {
        this.oAuth2ClientProvider = oAuth2ClientProvider;
    }

    /**
     * 获取OAuth2配置信息
     *
     * @return OAuth2配置信息
     */
    @GetMapping(value = "/setting")
    public ApiResult<OAuth2SettingRespVO> getSetting() {
        var clientConfig = oAuth2ClientProvider.getoAuth2ClientConfigBO();
        if (clientConfig == null) {
            return ApiResult.fail("未初始化认证服务端配置");
        }

        OAuth2SettingRespVO respVO = new OAuth2SettingRespVO();
        respVO.setClientId(clientConfig.getClientId());
        respVO.setAuthorizeEndpoint(clientConfig.getAuthorizeEndpoint());
        return ApiResult.ok(respVO);
    }

    /**
     * 授权码换取AccessToken
     *
     * @param clientId 客户端ID
     * @param code     授权码
     * @return AccessToken
     */
    @GetMapping(value = "/code2token")
    public ApiResult<OAuthToken> code2Token(@Param("clientId") @NotBlank(message = "客户端ID不能为空") String clientId
            , @Param("code") @NotBlank(message = "授权码不能为空") String code
            , @Param("redirectUri") @NotBlank(message = "重定向地址不能为空") String redirectUri
            , @Param("codeVerifier") String codeVerifier) {
        return oAuth2ClientProvider.code2AccessToken(clientId, code, redirectUri, codeVerifier);
    }
}
