package com.elitesland.fin.application.service.recorder;

import com.elitesland.fin.application.convert.recorder.RecOrderConvert;
import com.elitesland.fin.application.convert.rectype.RecTypeConvert;
import com.elitesland.fin.application.facade.param.recorder.RecOrderSaveParam;
import com.elitesland.fin.application.facade.vo.recorder.RecOrderVO;
import com.elitesland.fin.domain.entity.recorder.RecOrder;
import com.elitesland.fin.domain.param.recorder.RecOrderPageParam;
import com.elitesland.fin.domain.service.recorder.RecOrderDomainService;
import com.elitesland.fin.domain.service.rectype.RecTypeDomainService;
import com.elitesland.fin.dubbo.system.SystemRpcService;
import com.elitesland.fin.dubbo.workflow.WorkflowRpcService;
import com.elitesland.fin.infr.dto.recorder.RecOrderDTO;
import com.elitesland.fin.infr.dto.rectype.RecTypeDTO;
import com.elitesland.fin.application.service.workflow.WorkFlowDefKey;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.WorkflowConstant;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;

/**
 * @author zhiyu.he
 * @date 2022/4/13 11:42
 */
@Service
@RequiredArgsConstructor
public class RecOrderServiceImpl implements RecOrderService {

    private final RecOrderDomainService recOrderDomainService;

    private final RecTypeDomainService recTypeDomainService;
    private final SystemRpcService systemRpcService;
    private final WorkflowRpcService workflowRpcService;

    @SysCodeProc
    @Override
    public PagingVO<RecOrderVO> page(RecOrderPageParam param) {
        PagingVO<RecOrderDTO> page = recOrderDomainService.page(param);
        PagingVO<RecOrderVO> recOrderVOPagingVO = RecOrderConvert.INSTANCE.convertPage(page);
        recOrderVOPagingVO.getRecords().forEach(vo -> {
            if (vo.getVerAmt() != null && vo.getTotalAmt() != null) {
                vo.setUnVerAmt(vo.getTotalAmt().subtract(vo.getVerAmt()));
            }
        });
        return recOrderVOPagingVO;
    }

    @SysCodeProc
    @Override
    public RecOrderVO queryById(Long id) {
        RecOrderDTO recOrderDTO = recOrderDomainService.queryById(id, false);
        RecOrderVO recOrderVO = RecOrderConvert.INSTANCE.dtoToVo(recOrderDTO);
        return recOrderVO;
    }

    @SysCodeProc
    @Override
    public RecOrderVO queryDetailsById(Long id) {
        RecOrderDTO recOrderDTO = recOrderDomainService.queryById(id, true);
        RecOrderVO recOrderVO = RecOrderConvert.INSTANCE.dtoToVo(recOrderDTO);
        return recOrderVO;
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<List<Long>> deleteByIds(List<Long> ids) {
        List<Long> deleteByIds = recOrderDomainService.deleteByIds(ids);
        return ApiResult.ok(deleteByIds);
    }

    @Override
    public ApiResult<RecOrderVO> defaultValue() {
        RecTypeDTO recTypeDTO = recTypeDomainService.defaultValue();
        RecOrderVO recOrderVO = RecTypeConvert.INSTANCE.typeToOrder(recTypeDTO);
        if (recOrderVO == null) {
            return ApiResult.ok(new RecOrderVO());
        }
        return ApiResult.ok(recOrderVO);
    }

    @Transactional(rollbackFor = {Exception.class})
    @Override
    public ApiResult<Long> save(RecOrderSaveParam param) {
        RecOrder order = RecOrderConvert.INSTANCE.convertParam(param);
        Long id = recOrderDomainService.save(order);
        return ApiResult.ok(id);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW, rollbackFor = {Exception.class})
    public ApiResult<Long> submit(RecOrderSaveParam param) {
        Boolean auto = recTypeDomainService.getAuto(param.getRecTypeId());
        RecOrder order = RecOrderConvert.INSTANCE.convertParam(param);
        //如果收款单类型是自动审核  更新审核人和时间
        if (auto) {
            order.setAuditDate(LocalDateTime.now());
            order.setAuditUser("admin");
            SysUserDTO user = systemRpcService.getUserByName("admin");
            order.setAuditUserId(user.getId());
        }
        Long id = recOrderDomainService.submit(order, auto);
        if (!auto) {
            // 启动工作流
            startWorkFlow(order, id);
        }

        return ApiResult.ok(id);
    }

    private void startWorkFlow(RecOrder order, Long id) {
        if (order.getProcInstId() == null
                || WorkflowConstant.CAN_START_PROC_STATUSES.contains(order.getProcInstStatus())) {
            // 启动流程
            String procInstName = "收款单审核-" + order.getRecOrderNo();
            String procKey = WorkFlowDefKey.FIN_REC_ORDER.name();
            ProcessInfo processInfo = workflowRpcService.startProcess(procKey, procInstName, id.toString(), new HashMap<>());
            recOrderDomainService.updateWorkInfo(processInfo, id);
        }
    }

}
