package com.elitesland.fin.infr.repo.arorder;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.arorder.QArOrderDO;
import com.elitesland.fin.domain.param.arorder.ArOrderPageParam;
import com.elitesland.fin.domain.param.arorder.ArOrderParam;
import com.elitesland.fin.infr.dto.arorder.ArOrderDTO;
import com.elitesland.fin.infr.dto.common.ArVerDTO;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author wang.xl
 * @version V1.0
 * @Package com.elitesland.fin.infr.repo.aporder
 * @date 2022/3/16 10:38
 */
@Component
@RequiredArgsConstructor
public class ArOrderRepoProc {


    private final JPAQueryFactory jpaQueryFactory;
    private final QArOrderDO qArOrderDO = QArOrderDO.arOrderDO;


    public PagingVO<ArOrderDTO> page(ArOrderPageParam arOrderPageParam, List<Long> ids) {

        List<Predicate> predicates = new ArrayList<>();
        checkParam(arOrderPageParam, predicates);
        // 来源单据
        if (StringUtils.isNotBlank(arOrderPageParam.getCreateMode())) {
            predicates.add(qArOrderDO.createMode.eq(arOrderPageParam.getCreateMode()));
        }
        // 是否期初
        if (null != arOrderPageParam.getInitFlag()) {
            predicates.add(qArOrderDO.initFlag.eq(arOrderPageParam.getInitFlag()));
        }
        // 来源单号明细搜索
        if (StringUtils.isNotBlank(arOrderPageParam.getSourceNoDtl())) {
            if (CollUtil.isNotEmpty(ids)) {
                predicates.add(qArOrderDO.id.in(ids));
            } else {
                return new PagingVO<>(0, new ArrayList<>());
            }
        }
        // 销售组织
        buildPredicates(arOrderPageParam, predicates);

        predicates.add(qArOrderDO.deleteFlag.eq(0));
        JPAQuery<ArOrderDTO> query = select(ArOrderDTO.class)
                .where(ExpressionUtils.allOf(predicates));
        arOrderPageParam.setPaging(query);
        arOrderPageParam.fillOrders(query, qArOrderDO);
        return PagingVO.<ArOrderDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    private void buildPredicates(ArOrderPageParam arOrderPageParam, List<Predicate> predicates) {
        if (StringUtils.isNotBlank(arOrderPageParam.getOrgCode())) {
            predicates.add(qArOrderDO.orgCode.eq(arOrderPageParam.getOrgCode()));
        }
        if (null != arOrderPageParam.getOrgId()) {
            predicates.add(qArOrderDO.orgId.eq(arOrderPageParam.getOrgId()));
        }
        // 客户
        if (null != arOrderPageParam.getCustId()) {
            predicates.add(qArOrderDO.custId.eq(arOrderPageParam.getCustId()));
        }
        if (StringUtils.isNotBlank(arOrderPageParam.getCustName())) {
            predicates.add(qArOrderDO.custName.like("%" + arOrderPageParam.getCustName() + "%"));
        }
        if (StringUtils.isNotBlank(arOrderPageParam.getCurrCode())) {
            predicates.add(qArOrderDO.currCode.eq(arOrderPageParam.getCurrCode()));
        }
        // 销售业务员
        if (StringUtils.isNotBlank(arOrderPageParam.getSaleUser())) {
            predicates.add(qArOrderDO.saleUser.like("%" + arOrderPageParam.getSaleUser() + "%"));
        }
        // 销售部门
        if (null != arOrderPageParam.getBuId()) {
            predicates.add(qArOrderDO.buId.eq(arOrderPageParam.getBuId()));
        }
        if (StringUtils.isNotBlank(arOrderPageParam.getBuCode())) {
            predicates.add(qArOrderDO.buCode.eq(arOrderPageParam.getBuCode()));
        }
        // 审核日期
        if (null != arOrderPageParam.getAuditDateStart() && null != arOrderPageParam.getAuditDateEnd()) {
            predicates.add(qArOrderDO.auditDate.between(arOrderPageParam.getAuditDateStart(), arOrderPageParam.getAuditDateEnd()));
        }
    }

    private void checkParam(ArOrderPageParam arOrderPageParam, List<Predicate> predicates) {
        // 公司
        if (null != arOrderPageParam.getOuId()) {
            predicates.add(qArOrderDO.ouId.eq(arOrderPageParam.getOuId()));
        }
        // 币种
        if (StringUtils.isNotBlank(arOrderPageParam.getCurrCode())) {
            predicates.add(qArOrderDO.currCode.eq(arOrderPageParam.getCurrCode()));
        }
        if (StringUtils.isNotBlank(arOrderPageParam.getCurrName())) {
            predicates.add(qArOrderDO.currName.eq(arOrderPageParam.getCurrName()));
        }
        // 业务日期
        if (null != arOrderPageParam.getBuDateStart() && null != arOrderPageParam.getBuDateEnd()) {
            predicates.add(qArOrderDO.buDate.between(arOrderPageParam.getBuDateStart(), arOrderPageParam.getBuDateEnd()));
        }
        // 来源系统单号
        if (StringUtils.isNotBlank(arOrderPageParam.getSourceNo())) {
            predicates.add(qArOrderDO.sourceNo.like('%' + arOrderPageParam.getSourceNo() + '%'));
        }
        // 应收单类型
        if (null != arOrderPageParam.getArTypeId()) {
            predicates.add(qArOrderDO.arTypeId.eq(arOrderPageParam.getArTypeId()));
        }
        // 单据状态
        if (StringUtils.isNotBlank(arOrderPageParam.getOrderState())) {
            predicates.add(qArOrderDO.orderState.eq(arOrderPageParam.getOrderState()));
        }
        // 核销状态
        if (StringUtils.isNotBlank(arOrderPageParam.getVerState())) {
            predicates.add(qArOrderDO.verState.eq(arOrderPageParam.getVerState()));
        }
        // 应付单号
        if (StringUtils.isNotBlank(arOrderPageParam.getArOrderNo())) {
            predicates.add(qArOrderDO.arOrderNo.like('%' + arOrderPageParam.getArOrderNo() + '%'));
        }
        // 含税金额
        if (null != arOrderPageParam.getTotalAmtStart()) {
            predicates.add(qArOrderDO.totalAmt.goe(arOrderPageParam.getTotalAmtStart()));
        }
        if (null != arOrderPageParam.getTotalAmtEnd()) {
            predicates.add(qArOrderDO.totalAmt.loe(arOrderPageParam.getTotalAmtEnd()));
        }
        // 经办人
        if (StringUtils.isNotBlank(arOrderPageParam.getOperator())) {
            predicates.add(qArOrderDO.operator.like('%' + arOrderPageParam.getOperator() + '%'));
        }
        // 创建人
        if (StringUtils.isNotBlank(arOrderPageParam.getCreator())) {
            predicates.add(qArOrderDO.creator.like('%' + arOrderPageParam.getCreator() + '%'));
        }
        // 审核人
        if (StringUtils.isNotBlank(arOrderPageParam.getAuditUser())) {
            predicates.add(qArOrderDO.auditUser.like('%' + arOrderPageParam.getAuditUser() + '%'));
        }
    }

    public List<ArOrderDTO> queryByIds(List<Long> ids) {
        return select(ArOrderDTO.class)
                .where(qArOrderDO.id.in(ids))
                .fetch();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qArOrderDO.id,
                qArOrderDO.sourceNo,
                qArOrderDO.ouCode,
                qArOrderDO.ouId,
                qArOrderDO.ouName,
                qArOrderDO.arTypeId,
                qArOrderDO.arTypeName,
                qArOrderDO.arTypeCode,
                qArOrderDO.currCode,
                qArOrderDO.currName,
                qArOrderDO.operator,
                qArOrderDO.operUserId,
                qArOrderDO.buType,
                qArOrderDO.totalAmt,
                qArOrderDO.totalCurAmt,
                qArOrderDO.buDate,
                qArOrderDO.auditUserId,
                qArOrderDO.auditUser,
                qArOrderDO.auditDate,
                qArOrderDO.orderState,
                qArOrderDO.exchangeRate,
                qArOrderDO.initFlag,
                qArOrderDO.taxFlag,
                qArOrderDO.buId,
                qArOrderDO.buCode,
                qArOrderDO.buName,
                qArOrderDO.payMentName,
                qArOrderDO.payMentCode,
                qArOrderDO.payMentId,
                qArOrderDO.createTime,
                qArOrderDO.updater,
                qArOrderDO.modifyTime,
                qArOrderDO.exclTaxAmt,
                qArOrderDO.taxAmt,
                qArOrderDO.taxCurAmt,
                qArOrderDO.arOrderNo,
                qArOrderDO.creator,
                qArOrderDO.createMode,
                qArOrderDO.remark,
                qArOrderDO.localCurrCode,
                qArOrderDO.localCurrName,
                qArOrderDO.exclTaxCurAmt,
                qArOrderDO.verState,
                qArOrderDO.verAmt,
                qArOrderDO.orgId,
                qArOrderDO.orgCode,
                qArOrderDO.orgName,
                qArOrderDO.saleUserId,
                qArOrderDO.saleUser,
                qArOrderDO.arOrderType,
                qArOrderDO.custName,
                qArOrderDO.custId,
                qArOrderDO.custCode,
                qArOrderDO.auditDataVersion,
                qArOrderDO.procInstId,
                qArOrderDO.procInstStatus,
                qArOrderDO.docType,
                qArOrderDO.docType2,
                qArOrderDO.docCls,
                qArOrderDO.approvedTime,
                qArOrderDO.submitTime
        )).from(qArOrderDO);
    }

    public ArOrderDTO get(Long id) {
        ArOrderDTO arOrderDTO = select(ArOrderDTO.class)
                .where(qArOrderDO.id.eq(id))
                .fetchOne();
        return arOrderDTO;
    }

    public Long audit(List<Long> ids, String content, SysUserDTO user) {
        long res = jpaQueryFactory.update(qArOrderDO)
                .set(qArOrderDO.auditDate, LocalDateTime.now())
                .set(qArOrderDO.auditRejection, content)
                .set(qArOrderDO.auditUser, user.getUsername())
                .set(qArOrderDO.auditUserId, user.getId())
                .where(qArOrderDO.id.in(ids))
                .execute();
        return res;
    }

    public Boolean queryByArTypeId(Long id) {
        return jpaQueryFactory.select(qArOrderDO.id)
                .from(qArOrderDO)
                .where(qArOrderDO.arTypeId.eq(id))
                .fetchCount() > 0;
    }

    public Long del(List<Long> ids) {
        long res = jpaQueryFactory.delete(qArOrderDO)
                .where(qArOrderDO.id.in(ids))
                .execute();
        return res;
    }

    public ArVerDTO queryVerAmtById(Long id) {
        return jpaQueryFactory.select(Projections.bean(ArVerDTO.class,
                qArOrderDO.id,
                qArOrderDO.totalAmt,
                qArOrderDO.verAmt,
                qArOrderDO.auditDataVersion)).from(qArOrderDO).where(qArOrderDO.id.eq(id)).fetchOne();
    }

    public long updateVerAmt(ArVerDTO arVerDTO) {
        return jpaQueryFactory.update(qArOrderDO).set(qArOrderDO.verState, arVerDTO.getVerState())
                .set(qArOrderDO.verAmt, arVerDTO.getVerAmt())
                .set(qArOrderDO.auditDataVersion, arVerDTO.getAuditDataVersion() + 1)
                .where(qArOrderDO.id.eq(arVerDTO.getId())
                        .and(qArOrderDO.totalAmt.goe(arVerDTO.getVerAmt()))
                        .and(qArOrderDO.auditDataVersion.eq(arVerDTO.getAuditDataVersion())))
                .execute();
    }

    public List<ArOrderDTO> getArOrderList(ArOrderParam arOrderParam) {
        List<Predicate> predicates = new ArrayList<>();
        // 公司
        if (null != arOrderParam.getOuId()) {
            predicates.add(qArOrderDO.ouId.eq(arOrderParam.getOuId()));
        }
        if (arOrderParam.getCustId() != null) {
            predicates.add(qArOrderDO.custId.eq(arOrderParam.getCustId()));
        }
        // 币种
        if (CharSequenceUtil.isNotBlank(arOrderParam.getCurrCode())) {
            predicates.add(qArOrderDO.currCode.eq(arOrderParam.getCurrCode()));
        }
        // 业务日期
        if (null != arOrderParam.getBuDateStart() && null != arOrderParam.getBuDateEnd()) {
            predicates.add(qArOrderDO.buDate.between(arOrderParam.getBuDateStart(), arOrderParam.getBuDateEnd()));
        }
        predicates.add(qArOrderDO.orderState.eq(UdcEnum.COM_APPLY_STATUS_COMPLETE.getValueCode()));
        List<ArOrderDTO> fetch = select(ArOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
        return fetch;
    }

    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qArOrderDO)
                .set(qArOrderDO.procInstStatus, ProcInstStatus.APPROVING)
                .set(qArOrderDO.procInstId, processInfo.getProcInstId())
                .set(qArOrderDO.submitTime, LocalDateTime.now())
                .where(qArOrderDO.id.eq(id));
        if (!Objects.equals(processInfo.getProcInstStatus(), ProcInstStatus.APPROVED)) {
            jpaUpdateClause.set(qArOrderDO.procInstStatus, ProcInstStatus.APPROVING);
        }
        jpaUpdateClause.execute();
    }
}
