package com.elitesland.fin.infr.repo.recorder;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.fin.common.UdcEnum;
import com.elitesland.fin.domain.entity.recorder.QRecOrderDO;
import com.elitesland.fin.domain.param.recorder.RecOrderPageParam;
import com.elitesland.fin.domain.param.recorder.RecOrderParam;
import com.elitesland.fin.infr.dto.common.ArVerDTO;
import com.elitesland.fin.infr.dto.recorder.RecOrderDTO;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author zhiyu.he
 * @date 2022/4/13 10:39
 */
@Component
@RequiredArgsConstructor
public class RecOrderRepoProc {
    private final JPAQueryFactory jpaQueryFactory;
    private final QRecOrderDO qRecOrderDO = QRecOrderDO.recOrderDO;

    public PagingVO<RecOrderDTO> page(RecOrderPageParam param, List<Long> ids) {
        // 条件拼接
        List<Predicate> predicates = new ArrayList<>();
        buildPredicates(param, ids, predicates);
        if (param.getReDateStart() != null && param.getReDateEnd() != null) {
            predicates.add(qRecOrderDO.reDate.between(param.getReDateStart(), param.getReDateEnd()));
        }
        if (param.getBuId() != null) {
            predicates.add(qRecOrderDO.buId.eq(param.getBuId()));
        }
        if (param.getOrgId() != null) {
            predicates.add(qRecOrderDO.orgId.eq(param.getOrgId()));
        }
        if (!StringUtils.isEmpty(param.getRecOrderNo())) {
            predicates.add(qRecOrderDO.recOrderNo.like("%" + param.getRecOrderNo() + "%"));
        }
        if (!StringUtils.isEmpty(param.getAuditUser())) {
            predicates.add(qRecOrderDO.auditUser.like("%" + param.getAuditUser() + "%"));
        }
        if (!StringUtils.isEmpty(param.getAuditRejection())) {
            predicates.add(qRecOrderDO.auditRejection.like("%" + param.getAuditRejection() + "%"));
        }
        if (param.getTotalAmtFrom() != null) {
            predicates.add(qRecOrderDO.totalAmt.goe(param.getTotalAmtFrom()));
        }
        if (param.getTotalAmtTo() != null) {
            predicates.add(qRecOrderDO.totalAmt.loe(param.getTotalAmtTo()));
        }
        if (param.getCustId() != null) {
            predicates.add(qRecOrderDO.custId.eq(param.getCustId()));
        }
        if (StringUtils.isNotBlank(param.getCustName())) {
            predicates.add(qRecOrderDO.custName.like("%" + param.getCustName() + "%"));
        }
        if (param.getRecTypeId() != null) {
            predicates.add(qRecOrderDO.recTypeId.eq(param.getRecTypeId()));
        }
        if (!StringUtils.isEmpty(param.getCreateMode())) {
            predicates.add(qRecOrderDO.createMode.eq(param.getCreateMode()));
        }
        predicates.add(qRecOrderDO.deleteFlag.eq(0));
        JPAQuery<RecOrderDTO> query = select(RecOrderDTO.class).where(ExpressionUtils.allOf(predicates));
        param.setPaging(query);
        param.fillOrders(query, qRecOrderDO);
        return PagingVO.<RecOrderDTO>builder()
                .total(query.fetchCount())
                .records(query.fetch())
                .build();
    }

    private void buildPredicates(RecOrderPageParam param, List<Long> ids, List<Predicate> predicates) {
        if (!CollUtil.isEmpty(ids)) {
            predicates.add(qRecOrderDO.id.in(ids));
        }
        //是否期初
        if (param.getInitFlag() != null) {
            predicates.add(qRecOrderDO.initFlag.eq(param.getInitFlag()));
        }
        if (param.getReFlag() != null) {
            predicates.add(qRecOrderDO.reFlag.eq(param.getReFlag()));
        }
        if (param.getAuditDateStart() != null && param.getAuditDateEnd() != null) {
            predicates.add(qRecOrderDO.auditDate.between(param.getAuditDateStart(), param.getAuditDateEnd()));
        }
        if (!StringUtils.isEmpty(param.getCurrCode())) {
            predicates.add(qRecOrderDO.currCode.eq((param.getCurrCode())));
        }
        if (!StringUtils.isEmpty(param.getSaleUser())) {
            predicates.add(qRecOrderDO.saleUser.like(("%" + param.getSaleUser() + "%")));
        }
        if (!StringUtils.isEmpty(param.getOrderState())) {
            predicates.add(qRecOrderDO.orderState.eq(param.getOrderState()));
        }
        if (param.getOuId() != null) {
            predicates.add(qRecOrderDO.ouId.eq(param.getOuId()));
        }
        if (param.getExchangeRate() != null) {
            predicates.add(qRecOrderDO.exchangeRate.eq(param.getExchangeRate()));
        }
        if (param.getRecOuId() != null) {
            predicates.add(qRecOrderDO.recOuId.eq(param.getRecOuId()));
        }
        if (!StringUtils.isEmpty(param.getVerState())) {
            predicates.add(qRecOrderDO.verState.eq(param.getVerState()));
        }
        if (param.getAuditDateEnd() != null && param.getAuditDateStart() != null) {
            predicates.add(qRecOrderDO.auditDate.between(param.getAuditDateStart(), param.getAuditDateEnd()));
        }
    }

    public RecOrderDTO queryById(Long id) {
        return select(RecOrderDTO.class)
                .where(qRecOrderDO.id.eq(id))
                .fetchOne();
    }

    public Long audit(List<Long> ids, String auditRejection, SysUserDTO user) {
        return jpaQueryFactory.update(qRecOrderDO)
                .set(qRecOrderDO.auditDate, LocalDateTime.now())
                .set(qRecOrderDO.auditRejection, auditRejection)
                .set(qRecOrderDO.auditUser, user.getUsername())
                .set(qRecOrderDO.auditUserId, user.getId())
                .where(qRecOrderDO.id.in(ids))
                .execute();
    }

    public List<RecOrderDTO> queryByIds(List<Long> ids) {
        return select(RecOrderDTO.class)
                .where(qRecOrderDO.id.in(ids))
                .fetch();
    }

    public void deleteById(List<Long> ids) {
        jpaQueryFactory.delete(qRecOrderDO)
                .where(qRecOrderDO.id.in(ids))
                .execute();
    }

    public List<RecOrderDTO> queryByRecTypeId(List<Long> recTypeId) {
        return select(RecOrderDTO.class)
                .where(qRecOrderDO.recTypeId.in(recTypeId))
                .fetch();
    }

    public List<RecOrderDTO> getRecOrderList(RecOrderParam recOrderParam) {
        List<Predicate> predicates = new ArrayList<>();
        // 公司
        if (null != recOrderParam.getOuId()) {
            predicates.add(qRecOrderDO.ouId.eq(recOrderParam.getOuId()));
        }
        if (recOrderParam.getCustId() != null) {
            predicates.add(qRecOrderDO.custId.eq(recOrderParam.getCustId()));
        }
        // 币种
        if (CharSequenceUtil.isNotBlank(recOrderParam.getCurrCode())) {
            predicates.add(qRecOrderDO.currCode.eq(recOrderParam.getCurrCode()));
        }
        // 业务日期
        if (null != recOrderParam.getBuDateStart() && null != recOrderParam.getBuDateEnd()) {
            predicates.add(qRecOrderDO.reDate.between(recOrderParam.getBuDateStart(), recOrderParam.getBuDateEnd()));
        }
        predicates.add(qRecOrderDO.orderState.eq(UdcEnum.COM_APPLY_STATUS_COMPLETE.getValueCode()));
        List<RecOrderDTO> fetch = select(RecOrderDTO.class).where(ExpressionUtils.allOf(predicates)).fetch();
        return fetch;
    }

    public ArVerDTO queryVerAmtById(Long id) {
        return jpaQueryFactory.select(Projections.bean(ArVerDTO.class,
                qRecOrderDO.id,
                qRecOrderDO.totalAmt,
                qRecOrderDO.verAmt,
                qRecOrderDO.auditDataVersion)).from(qRecOrderDO).where(qRecOrderDO.id.eq(id)).fetchOne();
    }

    public long updateVerAmt(ArVerDTO arVerDTO) {
        return jpaQueryFactory.update(qRecOrderDO).set(qRecOrderDO.verState, arVerDTO.getVerState())
                .set(qRecOrderDO.verAmt, arVerDTO.getVerAmt())
                .set(qRecOrderDO.auditDataVersion, arVerDTO.getAuditDataVersion() + 1)
                .where(qRecOrderDO.id.eq(arVerDTO.getId())
                        .and(qRecOrderDO.totalAmt.goe(arVerDTO.getVerAmt()))
                        .and(qRecOrderDO.auditDataVersion.eq(arVerDTO.getAuditDataVersion())))
                .execute();
    }

    private <T> JPAQuery<T> select(Class<T> cls) {
        return jpaQueryFactory.select(Projections.bean(cls,
                qRecOrderDO.id,
                qRecOrderDO.sourceNo,
                qRecOrderDO.ouCode,
                qRecOrderDO.ouId,
                qRecOrderDO.ouName,
                qRecOrderDO.arTypeId,
                qRecOrderDO.arTypeName,
                qRecOrderDO.arTypeCode,
                qRecOrderDO.recTypeId,
                qRecOrderDO.recTypeCode,
                qRecOrderDO.recTypeName,
                qRecOrderDO.recOrderNo,
                qRecOrderDO.currCode,
                qRecOrderDO.currName,
                qRecOrderDO.localCurrCode,
                qRecOrderDO.localCurrName,
                qRecOrderDO.totalAmt,
                qRecOrderDO.totalCurAmt,
                qRecOrderDO.auditUserId,
                qRecOrderDO.auditUser,
                qRecOrderDO.auditDate,
                qRecOrderDO.orderState,
                qRecOrderDO.exchangeRate,
                qRecOrderDO.initFlag,
                qRecOrderDO.realRecAmt,
                qRecOrderDO.realRecCurAmt,
                qRecOrderDO.reDate,
                qRecOrderDO.reFlag,
                qRecOrderDO.auditRejection,
                qRecOrderDO.createMode,
                qRecOrderDO.verState,
                qRecOrderDO.verAmt,
                qRecOrderDO.custId,
                qRecOrderDO.custCode,
                qRecOrderDO.custName,
                qRecOrderDO.buId,
                qRecOrderDO.buCode,
                qRecOrderDO.buName,
                qRecOrderDO.recOuCode,
                qRecOrderDO.recOuId,
                qRecOrderDO.recOuName,
                qRecOrderDO.orgCode,
                qRecOrderDO.orgId,
                qRecOrderDO.orgName,
                qRecOrderDO.taxAmt,
                qRecOrderDO.recOrderType,
                qRecOrderDO.saleUserId,
                qRecOrderDO.saleUser,
                qRecOrderDO.creator,
                qRecOrderDO.remark,
                qRecOrderDO.createUserId,
                qRecOrderDO.taxCurAmt,
                qRecOrderDO.createTime,
                qRecOrderDO.modifyTime,
                qRecOrderDO.updater,
                qRecOrderDO.auditDataVersion,
                qRecOrderDO.procInstId,
                qRecOrderDO.procInstStatus,
                qRecOrderDO.approvedTime,
                qRecOrderDO.submitTime,
                qRecOrderDO.addrNo,
                qRecOrderDO.suppAddrNo,
                qRecOrderDO.docType,
                qRecOrderDO.docType2,
                qRecOrderDO.docCls)
        ).from(qRecOrderDO);
    }

    public void updateWorkInfo(ProcessInfo processInfo, Long id) {
        JPAUpdateClause jpaUpdateClause = jpaQueryFactory.update(qRecOrderDO)
                .set(qRecOrderDO.procInstStatus, ProcInstStatus.APPROVING)
                .set(qRecOrderDO.procInstId, processInfo.getProcInstId())
                .set(qRecOrderDO.submitTime, LocalDateTime.now())
                .where(qRecOrderDO.id.eq(id));
        if (!Objects.equals(processInfo.getProcInstStatus(), ProcInstStatus.APPROVED)) {
            jpaUpdateClause.set(qRecOrderDO.procInstStatus, ProcInstStatus.APPROVING);
        }
        jpaUpdateClause.execute();
    }
}