package com.elitesland.tw.tw5.server.common.funConfig.util;

import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.template.*;
import com.elitesland.tw.tw5.api.common.funConfig.vo.BusinessTableFieldsVO;
import com.elitesland.tw.tw5.api.common.funConfig.vo.BusinessTableVO;
import com.elitesland.tw.tw5.server.common.StringUtil;
import com.elitesland.tw.tw5.server.common.util.FileUtil;
import org.springframework.util.ObjectUtils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 代码生成工具类
 *
 * @author duwh
 * @date 2023/06/20
 */
public class CodeGenUtil {

    private static final String TIMESTAMP = "Timestamp";
    private static final String LOCALDATE = "LocalDate";
    private static final String LOCALDATETIME = "LocalDateTime";

    private static final String BIGDECIMAL = "BigDecimal";

    public static final String PK = "PRI";

    public static final String EXTRA = "auto_increment";

    /**
     * 代码生成器
     *
     * @param columnInfos   表列属性配置
     * @param genConfig     表配置
     * @param projectModule 项目模块 tw crm pms
     * @param apiGorupName  apifox分组名称
     * @param packageName   包名
     * @param author        作者
     * @param moduleName    模块名称
     * @param prefix        去除表格前缀 前缀
     * @param cover         是否覆盖 已生成的文件
     * @throws IOException IOException
     */
    public static void generatorCode(List<BusinessTableFieldsVO> columnInfos, BusinessTableVO genConfig,
                                     String projectModule,
                                     String apiGorupName,
                                     String packageName,
                                     String author,
                                     String moduleName,
                                     String prefix,
                                     boolean cover
    ) throws IOException {
        Map<String, Object> genMap = getGenMap(columnInfos, genConfig, apiGorupName, packageName, author, moduleName, prefix);
        TemplateEngine engine = TemplateUtil.createEngine(new TemplateConfig("template", TemplateConfig.ResourceMode.CLASSPATH));
        // 生成后端代码
        List<String> templates = getAdminTemplateNames();
        String type = projectModule;
        if (StringUtil.isBlank(type)) {
            type = CodeGenDomainEnum.TW.getCode();
        }
        for (String templateName : templates) {
            Template template = engine.getTemplate("generator/" + type + "/" + templateName + ".ftl");
            String rootPath = System.getProperty("user.dir");
            String filePath = getAdminFilePath(templateName, moduleName, packageName, genMap.get("className").toString(), rootPath);

            assert filePath != null;
            File file = new File(filePath);

            // 如果非覆盖生成
            if (!cover && FileUtil.exist(file)) {
                continue;
            }
            // 生成代码
            genFile(file, template, genMap);
        }

        // 生成前端代码
        //templates = getFrontTemplateNames();
        //for (String templateName : templates) {
        //    Template template = engine.getTemplate("generator/front/" + templateName + ".ftl");
        //    String filePath = getFrontFilePath(templateName, genConfig.getApiPath(), genConfig.getPath(), genMap.get("changeClassName").toString());
        //
        //    assert filePath != null;
        //    File file = new File(filePath);
        //
        //    // 如果非覆盖生成
        //    if (!genConfig.getCover() && FileUtil.exist(file)) {
        //        continue;
        //    }
        //    // 生成代码
        //    genFile(file, template, genMap);
        //}
    }

    // 获取模版数据
    private static Map<String, Object> getGenMap(List<BusinessTableFieldsVO> columnInfos,
                                                 BusinessTableVO genConfig,
                                                 String apiGorupName,
                                                 String packageName,
                                                 String author,
                                                 String moduleName,
                                                 String prefix
    ) {
        // 存储模版字段数据
        Map<String, Object> genMap = new HashMap<>(16);
        // 接口别名
        genMap.put("apiAlias", genConfig.getName());
        genMap.put("apiGroupName", ObjectUtils.isEmpty(apiGorupName) ? "" : apiGorupName);
        // 包名称
        genMap.put("package", packageName);
        // 模块名称
        genMap.put("moduleName", ObjectUtils.isEmpty(moduleName) ? "" : "moduleName");
        // 作者
        genMap.put("author", ObjectUtils.isEmpty(author) ? "" : "codegen");
        // 创建日期
        genMap.put("date", LocalDate.now().toString());
        // 表名
        genMap.put("tableName", genConfig.getTableName());
        // 大写开头的类名
        String className = StringUtil.toCapitalizeCamelCase(genConfig.getTableName());
        // 小写开头的类名
        String changeClassName = StringUtil.toCamelCase(genConfig.getTableName());
        // 判断是否去除表前缀
        if (StringUtil.isNotEmpty(prefix)) {
            className = StringUtil.toCapitalizeCamelCase(StrUtil.removePrefix(genConfig.getTableName(), prefix));
            changeClassName = StringUtil.toCamelCase(StrUtil.removePrefix(genConfig.getTableName(), prefix));
            changeClassName = StringUtil.uncapitalize(changeClassName);
        }
        // 保存类名
        genMap.put("className", className);
        // 保存小写开头的类名
        genMap.put("changeClassName", changeClassName);
        // 存在 Timestamp 字段
        genMap.put("hasTimestamp", false);
        genMap.put("hasLocalDate", false);
        genMap.put("hasLocalDateTime", false);
        // 查询类中存在 Timestamp 字段
        genMap.put("queryHasTimestamp", false);
        // 存在 BigDecimal 字段
        genMap.put("hasBigDecimal", false);
        // 查询类中存在 BigDecimal 字段
        genMap.put("queryHasBigDecimal", false);
        // 是否需要创建查询
        genMap.put("hasQuery", false);
        // 自增主键
        genMap.put("auto", false);
        // 存在字典
        genMap.put("hasDict", false);
        // 存在日期注解
        genMap.put("hasDateAnnotation", false);
        // 保存字段信息
        List<Map<String, Object>> columns = new ArrayList<>();
        // 保存查询字段的信息
        List<Map<String, Object>> queryColumns = new ArrayList<>();
        // 存储字典信息
        List<String> dicts = new ArrayList<>();
        // 存储 between 信息
        List<Map<String, Object>> betweens = new ArrayList<>();
        // 存储不为空的字段信息
        List<Map<String, Object>> isNotNullColumns = new ArrayList<>();

        for (BusinessTableFieldsVO column : columnInfos) {
            Map<String, Object> listMap = new HashMap<>(16);
            // 字段描述
            listMap.put("remark", column.getRemark());
            // 字段类型
            listMap.put("columnKey", column.getType());
            // 主键类型
            String colType = ColUtil.cloToJava(column.getType());
            // 小写开头的字段名
            String changeColumnName = StringUtil.toCamelCase(column.getField());
            // 大写开头的字段名
            String capitalColumnName = StringUtil.toCapitalizeCamelCase(column.getField());
            if ("PK".equals(column.getExt2())) {
                // 存储主键类型
                genMap.put("pkColumnType", colType);
                // 存储小写开头的字段名
                genMap.put("pkChangeColName", changeColumnName);
                // 存储大写开头的字段名
                genMap.put("pkCapitalColName", capitalColumnName);
            }
            // 是否存在 Timestamp 类型的字段
            if (TIMESTAMP.equals(colType)) {
                genMap.put("hasTimestamp", true);
            }
            if (LOCALDATE.equals(colType)) {
                genMap.put("hasLocalDate", true);
            }
            if (LOCALDATETIME.equals(colType)) {
                genMap.put("hasLocalDateTime", true);
            }
            // 是否存在 BigDecimal 类型的字段
            if (BIGDECIMAL.equals(colType)) {
                genMap.put("hasBigDecimal", true);
            }
            // 主键是否自增
            if (EXTRA.equals(column.getExt1())) {
                genMap.put("auto", true);
            }
            // 主键存在字典
//            if (StringUtil.isNotBlank(column.getDictName())) {
//                genMap.put("hasDict", true);
//                dicts.add(column.getDictName());
//            }

            // 存储字段类型
            listMap.put("columnType", colType);
            // 存储字原始段名称
            listMap.put("columnName", column.getField());
            // 不为空
            listMap.put("istNotNull", column.getNotNull());
            // 字段列表显示
//            listMap.put("columnShow", column.getListShow());
//            // 表单显示
//            listMap.put("formShow", column.getFormShow());
            // 表单组件类型
//            listMap.put("formType", StringUtil.isNotBlank(column.getFormType()) ? column.getFormType() : "Input");
            // 小写开头的字段名称
            listMap.put("changeColumnName", changeColumnName);
            //大写开头的字段名称
            listMap.put("capitalColumnName", capitalColumnName);
            // 字典名称
//            listMap.put("dictName", column.getDictName());
            // 日期注解
//            listMap.put("dateAnnotation", column.getDateAnnotation());
//            if (StringUtils.isNotBlank(column.getDateAnnotation())) {
//                genMap.put("hasDateAnnotation", true);
//            }
            // 添加非空字段信息
            if (column.getNotNull()) {
                isNotNullColumns.add(listMap);
            }
            // 判断是否有查询，如有则把查询的字段set进columnQuery
            //TODO 查询方式 默认 =
            if (!StringUtil.isBlank("=")) {
                // 查询类型
                listMap.put("queryType", "=");
                // 是否存在查询
                genMap.put("hasQuery", true);
                if (TIMESTAMP.equals(colType)) {
                    // 查询中存储 Timestamp 类型
                    genMap.put("queryHasTimestamp", true);
                }
                if (BIGDECIMAL.equals(colType)) {
                    // 查询中存储 BigDecimal 类型
                    genMap.put("queryHasBigDecimal", true);
                }
                if ("between".equalsIgnoreCase("=")) {
                    betweens.add(listMap);
                } else {
                    // 添加到查询列表中
                    queryColumns.add(listMap);
                }
            }
            // 主键 不添加
            if (PK.equals(column.getType())) {
                continue;
            }
            columns.add(listMap);
        }
        // 保存字段列表
        genMap.put("columns", columns);
        // 保存查询列表
        genMap.put("queryColumns", queryColumns);
        // 保存字段列表
        genMap.put("dicts", dicts);
        // 保存查询列表
        genMap.put("betweens", betweens);
        // 保存非空字段信息
        genMap.put("isNotNullColumns", isNotNullColumns);
        return genMap;
    }

    /**
     * 获取后端代码模板名称
     *
     * @return List
     */
    private static List<String> getAdminTemplateNames() {
        List<String> templateNames = new ArrayList<>();
        //templateNames.add("Entity");
        //templateNames.add("Dto");
        //templateNames.add("Mapper");
        //templateNames.add("Controller");
        //templateNames.add("QueryCriteria");
        //templateNames.add("Service");
        //templateNames.add("ServiceImpl");
        //templateNames.add("Repository");

        templateNames.add("Controller");
        templateNames.add("Convert");
        templateNames.add("DO");
        templateNames.add("DAO");
        templateNames.add("Payload");
        templateNames.add("Query");
        templateNames.add("Repo");
        templateNames.add("Service");
        templateNames.add("ServiceImpl");
        templateNames.add("VO");
        return templateNames;
    }

    /**
     * 定义后端文件路径以及名称
     */
    private static String getAdminFilePath(String templateName, String moduleName, String packageName, String className, String rootPath) {
        String projectPath = rootPath + File.separator + moduleName;
        String packagePath = projectPath + File.separator + "src" + File.separator + "main" + File.separator + "java" + File.separator;
        if (!ObjectUtils.isEmpty(packageName)) {
            packagePath += packageName.replace(".", File.separator) + File.separator;
        }
        // api层代码
        if ("Service".equals(templateName)) {
            return packagePath + "service" + File.separator + className + "Service.java";
        }
        if ("Payload".equals(templateName)) {
            return packagePath + "payload" + File.separator + className + "Payload.java";
        }
        if ("VO".equals(templateName)) {
            return packagePath + "vo" + File.separator + className + "VO.java";
        }
        if ("Query".equals(templateName)) {
            return packagePath + "query" + File.separator + className + "Query.java";
        }

        if ("Controller".equals(templateName)) {
            return packagePath + "controller" + File.separator + className + "Controller.java";
        }

        // server层代码
        if ("Entity".equals(templateName)) {
            return packagePath + "domain" + File.separator + className + ".java";
        }

        if ("ServiceImpl".equals(templateName)) {
            return packagePath + "service" + File.separator + className + "ServiceImpl.java";
        }
        if ("DO".equals(templateName)) {
            return packagePath + "entity" + File.separator + className + "DO.java";
        }
        if ("DAO".equals(templateName)) {
            return packagePath + "dao" + File.separator + className + "DAO.java";
        }

        if ("Convert".equals(templateName)) {
            return packagePath + "convert" + File.separator + className + "Convert.java";
        }
        if ("Repo".equals(templateName)) {
            return packagePath + "repo" + File.separator + className + "Repo.java";
        }

        if ("QueryCriteria".equals(templateName)) {
            return packagePath + "service" + File.separator + "dto" + File.separator + className + "QueryCriteria.java";
        }

        if ("Mapper".equals(templateName)) {
            return packagePath + "service" + File.separator + "mapstruct" + File.separator + className + "Mapper.java";
        }

        if ("Repository".equals(templateName)) {
            return packagePath + "repository" + File.separator + className + "Repository.java";
        }

        return null;
    }


    private static void genFile(File file, Template template, Map<String, Object> map) throws IOException {
        // 生成目标文件
        Writer writer = null;
        try {
            FileUtil.touch(file);
            writer = new FileWriter(file);
            template.render(map, writer);
        } catch (TemplateException | IOException e) {
            throw new RuntimeException(e);
        } finally {
            assert writer != null;
            writer.close();
        }
    }

}
