package com.elitesland.tw.tw5.server.common.util;

import com.elitescloud.cloudt.common.base.QBaseModel;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.tw.tw5.api.common.TwQueryParam;
import com.elitesland.tw.tw5.server.common.TwException;
import com.querydsl.core.types.Order;
import com.querydsl.core.types.OrderSpecifier;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAUpdateClause;
import org.springframework.util.ObjectUtils;

import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * sql工具类
 *
 * @author zhangyongqiang
 */
public class SqlUtilPlus {

    public static String toSqlLikeString(String value) {
        return "%" + value + "%";
    }


    /**
     * 分页排序工具类，动态处理分页字段。 当pathBase参数有多个对象时，且排序字段存在多个表时，以靠前的Q对象优先
     *
     * @param orders   排序数据
     * @param pathBase Q对象数组
     * @return 排序order
     */
    @SuppressWarnings("rawtypes,unchecked")
    public static OrderSpecifier<?>[] getSortedColumn(List<OrderItem> orders, EntityPathBase... pathBase) {
        if (orders == null || orders.size() == 0) {
            Path<String> fieldPath = Expressions.path(String.class, pathBase[0], "createTime");
            return new OrderSpecifier[]{new OrderSpecifier<>(Order.DESC, fieldPath)};
        } else {
            OrderSpecifier<String>[] orderSpecifiers = new OrderSpecifier[orders.size()];
            for (int i = 0; i < orders.size(); i++) {
                OrderItem orderItem = orders.get(i);
                Order order = orderItem.isAsc() ? Order.ASC : Order.DESC;
                EntityPathBase ownerEntity = getOwnerEntity(orderItem.getColumn(), pathBase);
                if (ownerEntity == null) {
                    throw TwException.error("", "不存在要排序的字段");
                }
                Path<String> fieldPath = Expressions.path(String.class, ownerEntity, orderItem.getColumn());
                orderSpecifiers[i] = new OrderSpecifier<>(order, fieldPath);
            }
            return orderSpecifiers;
        }
    }


    // 存储Q对象名称与相对应字段
    private static final Map<String, List<String>> fieldMap = new HashMap<>();

    /**
     * 获得字段属于哪个Q对象
     *
     * @param column   字段名称
     * @param pathBase 多个Q对象
     * @return Q对象
     */
    @SuppressWarnings("rawtypes")
    private static EntityPathBase getOwnerEntity(String column, EntityPathBase... pathBase) {
        EntityPathBase result = null;
        for (EntityPathBase path : pathBase) {
            String className = path.getClass().getName();
            List<String> stringFields = fieldMap.get(className);
            if (stringFields == null) {
                Field[] declaredFields = path.getClass().getDeclaredFields();
                List<String> stringList = new ArrayList<>(declaredFields.length);
                for (Field declaredField : declaredFields) {
                    if (Path.class.isAssignableFrom(declaredField.getType())) {
                        stringList.add(declaredField.getName());
                    }
                }
                stringFields = stringList;
                fieldMap.put(className, stringList);
            }
            // 当前Q对象是否包含字段
            if (stringFields.contains(column)) {
                result = path;
                break;
            }
        }
        return result;
    }

    /**
     * 常用基础查询条件拼装
     *
     * @param jpaQuery   jpaQuery对象
     * @param qBaseModel qBaseModel
     * @param query      查询条件
     * @param <T>        范型对象
     */
    public static <T> void handleCommonJpaQuery(JPAQuery<T> jpaQuery, QBaseModel qBaseModel, TwQueryParam query) {
        jpaQuery.where(qBaseModel.deleteFlag.eq(0));
        if (!ObjectUtils.isEmpty(query.getId())) {
            jpaQuery.where(qBaseModel.id.eq(query.getId()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            jpaQuery.where(qBaseModel.createUserId.eq(query.getCreateUserId()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateTimeStart())) {
            jpaQuery.where(qBaseModel.createTime.goe(query.getCreateTimeStart()));
        }
        if (!ObjectUtils.isEmpty(query.getCreateTimeEnd())) {
            jpaQuery.where(qBaseModel.createTime.loe(query.getCreateTimeEnd()));
        }
        if (!ObjectUtils.isEmpty(query.getRemark())) {
            jpaQuery.where(qBaseModel.remark.like(SqlUtilPlus.toSqlLikeString(query.getRemark())));
        }
    }

    /**
     * 必要的更新字段
     *
     * @param update     更新语句
     * @param qBaseModel 更新对象
     */
    public static void updateCommonJpaQuery(JPAUpdateClause update, QBaseModel qBaseModel) {
        SysUserDTO userDTO = AuthContextUtil.getLoginUser();
        update.set(qBaseModel.modifyUserId, userDTO.getId());
        update.set(qBaseModel.updater, userDTO.getFirstName());
        update.set(qBaseModel.modifyTime, LocalDateTime.now());
    }
}
