package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitesland.tw.tw5.api.prd.my.query.PrdUserMessageQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.PrdUserMessageTypeVO;
import com.elitesland.tw.tw5.api.prd.my.vo.PrdUserMessageVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QPrdUserMessageDO;
import com.elitesland.tw.tw5.server.prd.system.entity.QPrdMessageConfigDO;
import com.elitesland.tw.tw5.server.prd.my.entity.PrdUserMessageDO;
import com.elitesland.tw.tw5.server.prd.my.repo.PrdUserMessageRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Expression;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QTuple;
import com.querydsl.core.types.SubQueryExpression;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.hibernate.criterion.SubqueryExpression;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 系统选择项dao
 *
 * @author wangding
 */
@Repository
@RequiredArgsConstructor
public class PrdUserMessageDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdUserMessageRepo repo;

    private final QPrdUserMessageDO qdo = QPrdUserMessageDO.prdUserMessageDO;
    private final QPrdMessageConfigDO qdoConfig = QPrdMessageConfigDO.prdMessageConfigDO;

    /**
     * 查询组织关系
     *
     * @param userId
     * @return 结果
     */
    public Long queryTotal(Long userId) {
        JPAQuery<PrdUserMessageDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);//.leftJoin(qdoConfig).on(qdo.messageId.longValue().eq(qdoConfig.id.longValue()))
        jpaQuery.where(qdo.messageStatus.eq(0));
        jpaQuery.where(qdo.userId.eq(userId));
        return jpaQuery.fetchCount();
    }

    /**
     * 逻辑删除
     *
     * @param keys   主键
     * @param userId
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys, Long userId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.userId.eq(userId))
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdUserMessageVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdUserMessageVO.class,
                qdo.id,
                qdo.messageStatus,
                qdo.messageId,
                qdo.createSource,
                // qdo.messageCode,
                qdoConfig.objectId,
                qdo.messageTitle,
                qdo.messageContent,
                qdo.noticeScope,
                qdo.noticeSource,
                qdo.contentType,
                qdo.contentBigType,
                qdo.messageTag,
                qdo.remark,
                qdo.createUserId,
                qdo.createTime
        )).from(qdo).leftJoin(qdoConfig).on(qdo.messageId.longValue().eq(qdoConfig.id.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdUserMessageVO> getJpaQueryWhere(PrdUserMessageQuery query, Long userId) {
        JPAQuery<PrdUserMessageVO> jpaQuery = getJpaQuerySelect();

        jpaQuery.where(qdo.userId.eq(userId));
        if (!ObjectUtils.isEmpty(query.getContentBigType())) {
            jpaQuery.where(qdo.contentBigType.eq(query.getContentBigType()));
        }
        if (!ObjectUtils.isEmpty(query.getMessageStatus())) {
            jpaQuery.where(qdo.messageStatus.eq(query.getMessageStatus()));
        }
//        if (!ObjectUtils.isEmpty(query.getMessageCode())) {
//            jpaQuery.where(qdoConfig.messageCode.eq(query.getMessageCode()));
//        }
        if (!ObjectUtils.isEmpty(query.getMessageTitle())) {
            jpaQuery.where(qdo.messageTitle.like(SqlUtil.toSqlLikeString(query.getMessageTitle())));
        }
        if (!ObjectUtils.isEmpty(query.getMessageContent())) {
            jpaQuery.where(qdo.messageContent.like(SqlUtil.toSqlLikeString(query.getMessageContent())));
        }
        if (!ObjectUtils.isEmpty(query.getContentType())) {
            jpaQuery.where(qdo.contentType.eq(query.getContentType()));
        }
        if (!ObjectUtils.isEmpty(query.getMessageTag())) {
            jpaQuery.where(qdo.messageTag.eq(query.getMessageTag()));
        }

        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.asc("messageStatus"));
        orderse.add(OrderItem.asc("sortIndex"));
        orderse.add(OrderItem.desc("createTime"));
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        return jpaQuery;
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdUserMessageVO queryByKey(Long id) {
        JPAQuery<PrdUserMessageVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    /**
     * 消息已读
     *
     * @param keys 主键
     * @return 修改的行数
     */
    public long updateStatus(List<Long> keys, Long userId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.messageStatus, 1)
                .where(qdo.userId.eq(userId))
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdUserMessageVO> queryPaging(PrdUserMessageQuery query, Long userId) {
        JPAQuery<PrdUserMessageVO> jpaQuery = getJpaQueryWhere(query, userId);
        List<PrdUserMessageVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetch();
        return PagingVO.<PrdUserMessageVO>builder().records(result).total(jpaQuery.fetchCount()).build();
    }


    /**
     * 查询所有消息类型
     * @return
     */
    public List<PrdUserMessageTypeVO> findMessageTypeList() {
        JPAQuery<PrdUserMessageTypeVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PrdUserMessageTypeVO.class,
                        qdo.contentType
                )).from(qdo)
                .where(qdo.messageStatus.eq(0)
                        .and(qdo.userId.eq(GlobalUtil.getLoginUserId()))
                        .and(qdo.deleteFlag.eq(0)))
                .groupBy(qdo.contentType);
        return jpaQuery.fetch();
    }

    /**
     * 查询未读消息类型及数量
     * @return
     */
    public List<PrdUserMessageTypeVO> findUnreadMessageTypeList() {
        JPAQuery<PrdUserMessageTypeVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PrdUserMessageTypeVO.class,
                qdo.contentType,
                qdo.count().as("unreadContentCount")
                )).from(qdo)
                .where(qdo.messageStatus.eq(0)
                        .and(qdo.userId.eq(GlobalUtil.getLoginUserId()))
                        .and(qdo.deleteFlag.eq(0)
                        .and(qdo.messageStatus.eq(0))))
                .groupBy(qdo.contentType);
        return jpaQuery.fetch();
    }

    /**
     * 查询未读消息的最新一条消息
     * @return
     */
    public List<PrdUserMessageVO> findLastUserMessage() {
        JPAQuery<PrdUserMessageVO> jpaQuery = getJpaQuerySelect()
                .where(qdo.messageStatus.eq(0)
                        .and(qdo.userId.eq(GlobalUtil.getLoginUserId()))
                        .and(qdo.deleteFlag.eq(0))
                        .and(qdo.createTime
                                .in(JPAExpressions.select(qdo.createTime.max()).from(qdo)
                                .where(qdo.userId.eq(GlobalUtil.getLoginUserId())
                                .and(qdo.messageStatus.eq(0))
                                .and(qdo.deleteFlag.eq(0)))
                                .groupBy(qdo.contentType)))
                );
        return jpaQuery.fetch();
    }

    /**
     * 根据消息类别查询消息列表
     * @return result
     */
    public List<PrdUserMessageVO> findUnreadMessageListByType(String contentType) {
        JPAQuery<PrdUserMessageVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.userId.eq(GlobalUtil.getLoginUserId())
//                .and(qdo.messageStatus.eq(0)
                 .and(qdo.deleteFlag.eq(0)
                 .and(qdo.contentType.eq(contentType)))
        );
        jpaQuery.orderBy(qdo.createTime.desc(),qdo.messageStatus.desc());
        return jpaQuery.fetch();
    }
}
