package com.elitesland.tw.tw5.server.prd.shorturl.service;

import cn.hutool.core.util.RandomUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.core.common.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.shorturl.payload.ShortUrlRefPayload;
import com.elitesland.tw.tw5.api.prd.shorturl.query.ShortUrlConfQuery;
import com.elitesland.tw.tw5.api.prd.shorturl.query.ShortUrlRefQuery;
import com.elitesland.tw.tw5.api.prd.shorturl.service.ShortUrlConfService;
import com.elitesland.tw.tw5.api.prd.shorturl.service.ShortUrlRefService;
import com.elitesland.tw.tw5.api.prd.shorturl.vo.ShortUrlConfVO;
import com.elitesland.tw.tw5.api.prd.shorturl.vo.ShortUrlRefVO;
import com.elitesland.tw.tw5.server.common.util.URLShortenerUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.shorturl.convert.ShortUrlRefConvert;
import com.elitesland.tw.tw5.server.prd.shorturl.dao.ShortUrlRefDAO;
import com.elitesland.tw.tw5.server.prd.shorturl.entity.ShortUrlRefDO;
import com.elitesland.tw.tw5.server.prd.shorturl.repo.ShortUrlRefRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Optional;

/**
 * 短链服务关系表
 *
 * @author duwh
 * @date 2023-05-19
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ShortUrlRefServiceImpl extends BaseServiceImpl implements ShortUrlRefService {

    private final ShortUrlRefRepo shortUrlRefRepo;
    private final ShortUrlRefDAO shortUrlRefDAO;
    @Autowired
    private ShortUrlConfService shortUrlConfService;

    @Override
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public PagingVO<ShortUrlRefVO> queryPaging(ShortUrlRefQuery query) {
        return shortUrlRefDAO.queryPaging(query);
    }

    @Override
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public List<ShortUrlRefVO> queryListDynamic(ShortUrlRefQuery query) {
        return shortUrlRefDAO.queryListDynamic(query);
    }

    @Override
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO queryByKey(Long key) {
        ShortUrlRefDO entity = shortUrlRefRepo.findById(key).orElseGet(ShortUrlRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ShortUrlRefVO vo = ShortUrlRefConvert.INSTANCE.toVo(entity);
        return vo;
    }

    @Override
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO queryByCodeAndAddVisitNum(String code) {
        if (StringUtils.hasText(code)) {
            ShortUrlRefVO shortUrlRefVO = shortUrlRefDAO.queryByCode(code.trim());
            if (null != shortUrlRefVO) {
                Integer visitNum = shortUrlRefVO.getVisitNum();
                if (null == visitNum) {
                    visitNum = 0;
                }
                ShortUrlRefPayload shortUrlRefPayload = new ShortUrlRefPayload();
                shortUrlRefPayload.setId(shortUrlRefVO.getId());
                shortUrlRefPayload.setVisitNum(visitNum + 1);
                shortUrlRefDAO.updateByKeyDynamicNotLogin(shortUrlRefPayload);
            }
            return shortUrlRefVO;
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO insert(ShortUrlRefPayload payload) {
        ShortUrlRefDO entityDo = ShortUrlRefConvert.INSTANCE.toDo(payload);
        return ShortUrlRefConvert.INSTANCE.toVo(shortUrlRefRepo.save(entityDo));
    }

    /**
     * 生成短链-重定向策略
     *
     * @param longUrlSuffix 长url后缀
     * @return {@link ShortUrlRefVO}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO saveRedirect(String longUrlSuffix) {
        ShortUrlConfVO shortUrlConfVO = new ShortUrlConfVO();
        shortUrlConfVO.setShortUrlRelative("/tw-server5-crm/s/r/");
        shortUrlConfVO.setShortUrlPre("https://crm.tw.elitesland.com/tw-server5-crm/s/r/");
        shortUrlConfVO.setLongUrlPre("https://crmapp.tw.elitesland.com/");
//        shortUrlConfVO.setShortUrlPre("http://localhost:8000/s/r/");
//        shortUrlConfVO.setLongUrlPre("http://localhost:8000");
        return commonSave(longUrlSuffix, shortUrlConfVO);
    }

    /**
     * 生成短链-重定向策略
     *
     * @param longUrlSuffix 长url后缀
     * @return {@link ShortUrlRefVO}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO saveRedirect(String longUrlSuffix, String shortUrlRelative, String shortUrlPre, String longUrlPre) {
        ShortUrlConfVO shortUrlConfVO = new ShortUrlConfVO();
        shortUrlConfVO.setShortUrlRelative("/tw-server5-crm/s/r/");
        if (StringUtils.hasText(shortUrlRelative)) {
            shortUrlConfVO.setShortUrlRelative(shortUrlRelative);
        }
        shortUrlConfVO.setShortUrlPre("https://crm.tw.elitesland.com/tw-server5-crm/s/r/");
        if (StringUtils.hasText(shortUrlPre)) {
            shortUrlConfVO.setShortUrlPre(shortUrlPre);
        }
        shortUrlConfVO.setLongUrlPre("https://crmapp.tw.elitesland.com/");
        if (StringUtils.hasText(longUrlPre)) {
            shortUrlConfVO.setLongUrlPre(longUrlPre);
        }
        return commonSave(longUrlSuffix, shortUrlConfVO);
    }

    /**
     * 生成短链-转发策略
     *
     * @param longUrlSuffix 长url后缀
     * @return {@link ShortUrlRefVO}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO saveForward(String longUrlSuffix) {
        ShortUrlConfVO shortUrlConfVO = new ShortUrlConfVO();
        shortUrlConfVO.setShortUrlPre("https://crm.tw.elitesland.com/tw-server5-crm/s/f/");
        shortUrlConfVO.setLongUrlPre("https://crm.tw.elitesland.com/");
//        shortUrlConfVO.setShortUrlPre("http://localhost:8000/s/f/");
//        shortUrlConfVO.setLongUrlPre("http://localhost:8000");
        return commonSave(longUrlSuffix, shortUrlConfVO);

    }

    @NotNull
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    private ShortUrlRefVO commonSave(String longUrlSuffix, ShortUrlConfVO shortUrlConfVO) {
        Long tenantId = GlobalUtil.getTenantId();
        // 查找短链配置 理论， 一个租户只能有一个配置
        ShortUrlConfQuery shortUrlConfQuery = new ShortUrlConfQuery();
        shortUrlConfQuery.setSecTenantId(tenantId);
        List<ShortUrlConfVO> shortUrlConfVOS = shortUrlConfService.queryListDynamic(shortUrlConfQuery);
        if (!CollectionUtils.isEmpty(shortUrlConfVOS)) {
            shortUrlConfVO = shortUrlConfVOS.get(0);
        }
        String shortUrlPre = shortUrlConfVO.getShortUrlPre();
        URLShortenerUtil urlShortenerUtil = new URLShortenerUtil(shortUrlPre);
        String longUrl = shortUrlConfVO.getLongUrlPre() + longUrlSuffix;
        String code = urlShortenerUtil.longToShortUrl(longUrl, RandomUtil.randomInt(10000));
        String shortUrl = shortUrlPre + code;

        // 判断是否重复生成
        ShortUrlRefQuery countQuery = new ShortUrlRefQuery();
        countQuery.setShortUrl(shortUrl);
        countQuery.setCode(code);
        countQuery.setLongUrl(longUrl);
        List<ShortUrlRefVO> shortUrlRefVOS = shortUrlRefDAO.queryListDynamic(countQuery);
        if (!CollectionUtils.isEmpty(shortUrlRefVOS)) {
            return shortUrlRefVOS.get(0);
        }

        // 生成关系
        ShortUrlRefPayload shortUrlRefPayload = new ShortUrlRefPayload();
        shortUrlRefPayload.setShortUrl(shortUrl);
        shortUrlRefPayload.setCode(code);
        shortUrlRefPayload.setLongUrl(longUrl);
        ShortUrlRefDO entityDo = ShortUrlRefConvert.INSTANCE.toDo(shortUrlRefPayload);
        return ShortUrlRefConvert.INSTANCE.toVo(shortUrlRefRepo.save(entityDo));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public ShortUrlRefVO update(ShortUrlRefPayload payload) {
        ShortUrlRefDO entity = shortUrlRefRepo.findById(payload.getId()).orElseGet(ShortUrlRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        ShortUrlRefDO entityDo = ShortUrlRefConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return ShortUrlRefConvert.INSTANCE.toVo(shortUrlRefRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            keys.stream().forEach(id -> {
                Optional<ShortUrlRefDO> optional = shortUrlRefRepo.findById(id);
                if (!optional.isEmpty()) {
                    ShortUrlRefDO entity = optional.get();
                    entity.setDeleteFlag(1);
                    shortUrlRefRepo.save(entity);
                }
            });
        }
    }

}
