package com.elitesland.tw.tw5.server.tenant.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.api.tenant.payload.AuthTenantPayload;
import com.elitesland.tw.tw5.api.tenant.query.AuthTenantQuery;
import com.elitesland.tw.tw5.api.tenant.vo.AuthTenantVO;
import com.elitesland.tw.tw5.server.tenant.entity.AuthTenantDO;
import com.elitesland.tw.tw5.server.tenant.entity.QAuthTenantDO;
import com.elitesland.tw.tw5.server.tenant.repo.AuthTenantRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 租户授权
 *
 * @author duwh
 * @date 2023-04-02
 */
@Repository
@RequiredArgsConstructor
public class AuthTenantDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AuthTenantRepo repo;
    private final QAuthTenantDO qdo = QAuthTenantDO.authTenantDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AuthTenantVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AuthTenantVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 授权租户数据
                qdo.authTenantId,
                // 授权模块 UDC 下拉选择项
                qdo.authType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AuthTenantVO> getJpaQueryWhere(AuthTenantQuery query) {
        JPAQuery<AuthTenantVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
            }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AuthTenantQuery query) {
        long total = jpaQueryFactory
            .select(qdo.count())
            .from(qdo)
            .where(where(query))
            .fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AuthTenantQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 授权租户数据 精确 */
        if (!ObjectUtils.isEmpty(query.getAuthTenantId())) {
            list.add(qdo.authTenantId.eq(query.getAuthTenantId()));
        }
        /** 授权模块 UDC 下拉选择项 精确 */
        if (!ObjectUtils.isEmpty(query.getAuthType())) {
            list.add(qdo.authType.eq(query.getAuthType()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AuthTenantVO queryByKey(Long id) {
        JPAQuery<AuthTenantVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AuthTenantVO> queryListDynamic(AuthTenantQuery query) {
        JPAQuery<AuthTenantVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AuthTenantVO> queryPaging(AuthTenantQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<AuthTenantVO> jpaQuery = getJpaQueryWhere(query);
        List<AuthTenantVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AuthTenantVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AuthTenantDO save(AuthTenantDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AuthTenantDO> saveAll(List<AuthTenantDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AuthTenantPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 授权租户数据
        if (payload.getAuthTenantId() != null) {
            update.set(qdo.authTenantId, payload.getAuthTenantId());
        }
        // 授权模块 UDC 下拉选择项
        if (payload.getAuthType() != null) {
            update.set(qdo.authType, payload.getAuthType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 授权租户数据
            if (nullFields.contains("authTenantId")) {
                update.setNull(qdo.authTenantId);
            }
            // 授权模块 UDC 下拉选择项
            if (nullFields.contains("authType")) {
                update.setNull(qdo.authType);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

